/*
 * Asterisk -- An open source telephony toolkit.
 *
 * Copyright (C) 1999 - 2025, Sangoma Technologies Corporation
 *
 * Mark Spencer <markster@digium.com>
 *
 * See http://www.asterisk.org for more information about
 * the Asterisk project. Please do not directly contact
 * any of the maintainers of this project for assistance;
 * the project provides a web site, mailing lists and IRC
 * channels for your use.
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License Version 2. See the LICENSE file
 * at the top of the source tree.
 */


/* Doxygenified Copyright Header */
/*!
 * \mainpage Asterisk -- The Open Source Telephony Project
 *
 * \par Welcome
 *
 * This documentation created by the Doxygen project clearly explains the
 * internals of the Asterisk software. This documentation contains basic
 * examples, developer documentation, support information, and information
 * for upgrading.
 *
 * \section community Community
 * Asterisk is a big project and has a busy community. Look at the
 * resources for questions and stick around to help answer questions.
 * \li \ref asterisk_community_resources
 *
 * \par Developer Documentation for Asterisk
 *
 * This is the main developer documentation for Asterisk. It is
 * generated by running "make progdocs" from the Asterisk source tree.
 *
 * In addition to the information available on the Asterisk source code,
 * please see the appendices for information on coding guidelines,
 * release management, commit policies, and more.
 *
 * \arg \ref AsteriskArchitecture
 *
 * \par Additional documentation
 * \arg \ref Licensing
 * \arg \ref DevDoc
 * \arg \ref configuration_file
 * \arg \ref channel_drivers
 * \arg \ref applications
 *
 * \section copyright Copyright and Author
 *
 * Copyright (C) 1999 - 2025, Sangoma Technologies Corporation.
 * Asterisk is a <a href="https://cdn.sangoma.com/wp-content/uploads/Sangoma-Trademark-Policy.pdf">registered trademark</a>
 * of <a rel="nofollow" href="http://www.sangoma.com">Sangoma Technologies Corporation</a>.
 *
 * \author Mark Spencer <markster@digium.com>
 *
 * See http://www.asterisk.org for more information about
 * the Asterisk project. Please do not directly contact
 * any of the maintainers of this project for assistance;
 * the project provides a web site, mailing lists, and IRC
 * channels for your use.
 *
 */

/*!
 * \page asterisk_community_resources Asterisk Community Resources
 * \par Websites
 * \li https://www.asterisk.org Asterisk Homepage
 * \li https://docs.asterisk.org Asterisk documentation
 *
 * \par Mailing Lists
 * \par
 * All lists: http://lists.digium.com/mailman/listinfo
 * \li aadk-commits	SVN commits to the AADK repository
 * \li asterisk-addons-commits	SVN commits to the Asterisk addons project
 * \li asterisk-announce	[no description available]
 * \li asterisk-biz	Commercial and Business-Oriented Asterisk Discussion
 * \li Asterisk-BSD	Asterisk on BSD discussion
 * \li asterisk-bugs	[no description available]
 * \li asterisk-commits	SVN commits to the Asterisk project
 * \li asterisk-dev	Asterisk Developers Mailing List
 * \li asterisk-doc	Discussions regarding The Asterisk Documentation Project
 * \li asterisk-embedded	Asterisk Embedded Development
 * \li asterisk-gui	Asterisk GUI project discussion
 * \li asterisk-gui-commits	SVN commits to the Asterisk-GUI project
 * \li asterisk-ha-clustering	Asterisk High Availability and Clustering List - Non-Commercial Discussion
 * \li Asterisk-i18n	Discussion of Asterisk internationalization
 * \li asterisk-r2	[no description available]
 * \li asterisk-scf-commits	Commits to the Asterisk SCF project code repositories
 * \li asterisk-scf-committee	Asterisk SCF Steering Committee discussions
 * \li asterisk-scf-dev	Asterisk SCF Developers Mailing List
 * \li asterisk-scf-wiki-changes	Changes to the Asterisk SCF space on wiki.asterisk.org
 * \li asterisk-security	Asterisk Security Discussion
 * \li asterisk-speech-rec	Use of speech recognition in Asterisk
 * \li asterisk-ss7	[no description available]
 * \li asterisk-users	Asterisk Users Mailing List - Non-Commercial Discussion
 * \li asterisk-video	Development discussion of video media support in Asterisk
 * \li asterisk-wiki-changes	Changes to the Asterisk space on wiki.asterisk.org
 * \li asterisknow	AsteriskNOW Discussion
 * \li dahdi-commits	SVN commits to the DAHDI project
 * \li digium-announce	Digium Product Announcements
 * \li Dundi	Distributed Universal Number Discovery
 * \li libiax2-commits	SVN commits to the libiax2 project
 * \li libpri-commits	SVN commits to the libpri project
 * \li libss7-commits	SVN commits to the libss7 project
 * \li svn-commits	SVN commits to the Digium repositories
 * \li Test-results	Results from automated testing
 * \li thirdparty-commits	SVN commits to the Digium third-party software repository
 * \li zaptel-commits	SVN commits to the Zaptel project
 *
 * \par Forums
 * \li Forums are located at http://forums.asterisk.org/
 *
 * \par IRC
 * \par
 * Use https://libera.chat IRC server to connect with Asterisk
 * developers and users in realtime.
 *
 * \li \verbatim #asterisk \endverbatim Asterisk Users Room
 * \li \verbatim #asterisk-dev \endverbatim Asterisk Developers Room
 *
 * \par More
 * \par
 * If you would like to add a resource to this list please create an issue
 * on the issue tracker with a patch.
 */

/*! \file
 * \brief Top level source file for Asterisk - the Open Source PBX.
 *	Implementation of PBX core functions and CLI interface.
 */

/*! \li \ref asterisk.c uses the configuration file \ref asterisk.conf
 * \addtogroup configuration_file
 */

/*! \page asterisk.conf asterisk.conf
 * \verbinclude asterisk.conf.sample
 */

/*** MODULEINFO
	<support_level>core</support_level>
 ***/

#include "asterisk.h"

#include "asterisk/_private.h"

#undef sched_setscheduler
#undef setpriority
#include <sys/time.h>
#include <fcntl.h>
#include <signal.h>
#include <sched.h>
#include <sys/un.h>
#include <sys/wait.h>
#include <ctype.h>
#include <sys/resource.h>
#include <grp.h>
#include <pwd.h>
#include <sys/stat.h>
#if defined(HAVE_SYSINFO)
#include <sys/sysinfo.h>
#elif defined(HAVE_SYSCTL)
#include <sys/param.h>
#include <sys/sysctl.h>
#include <sys/vmmeter.h>
#if defined(__FreeBSD__) || defined(__DragonFly__)
#include <vm/vm_param.h>
#endif
#if defined(HAVE_SWAPCTL)
#include <sys/swap.h>
#endif
#endif
#include <regex.h>
#include <histedit.h>

#if defined(SOLARIS)
int daemon(int, int);  /* defined in libresolv of all places */
#include <sys/loadavg.h>
#endif

#ifdef linux
#include <sys/prctl.h>
#ifdef HAVE_CAP
#include <sys/capability.h>
#endif /* HAVE_CAP */
#endif /* linux */

/* we define here the variables so to better agree on the prototype */
#include "asterisk/paths.h"
#include "asterisk/network.h"
#include "asterisk/cli.h"
#include "asterisk/channel.h"
#include "asterisk/translate.h"
#include "asterisk/pickup.h"
#include "asterisk/acl.h"
#include "asterisk/ulaw.h"
#include "asterisk/alaw.h"
#include "asterisk/callerid.h"
#include "asterisk/image.h"
#include "asterisk/tdd.h"
#include "asterisk/term.h"
#include "asterisk/manager.h"
#include "asterisk/cdr.h"
#include "asterisk/pbx.h"
#include "asterisk/app.h"
#include "asterisk/mwi.h"
#include "asterisk/lock.h"
#include "asterisk/utils.h"
#include "asterisk/file.h"
#include "asterisk/io.h"
#include "asterisk/config.h"
#include "asterisk/ast_version.h"
#include "asterisk/linkedlists.h"
#include "asterisk/devicestate.h"
#include "asterisk/presencestate.h"
#include "asterisk/module.h"
#include "asterisk/buildinfo.h"
#include "asterisk/xmldoc.h"
#include "asterisk/poll-compat.h"
#include "asterisk/test.h"
#include "asterisk/rtp_engine.h"
#include "asterisk/format.h"
#include "asterisk/aoc.h"
#include "asterisk/uuid.h"
#include "asterisk/sorcery.h"
#include "asterisk/bucket.h"
#include "asterisk/stasis.h"
#include "asterisk/json.h"
#include "asterisk/stasis_endpoints.h"
#include "asterisk/stasis_system.h"
#include "asterisk/security_events.h"
#include "asterisk/endpoints.h"
#include "asterisk/codec.h"
#include "asterisk/format_cache.h"
#include "asterisk/media_cache.h"
#include "asterisk/astdb.h"
#include "asterisk/options.h"
#include "asterisk/utf8.h"

#include "../defaults.h"

/*** DOCUMENTATION
	<managerEvent language="en_US" name="FullyBooted">
		<managerEventInstance class="EVENT_FLAG_SYSTEM">
			<since>
				<version>12.0.0</version>
			</since>
			<synopsis>Raised when all Asterisk initialization procedures have finished.</synopsis>
			<syntax>
				<parameter name="Status">
					<para>Informational message</para>
				</parameter>
				<parameter name="Uptime">
					<para>Seconds since start</para>
				</parameter>
				<parameter name="LastReload">
					<para>Seconds since last reload</para>
				</parameter>
			</syntax>
		</managerEventInstance>
	</managerEvent>
	<managerEvent language="en_US" name="Shutdown">
		<managerEventInstance class="EVENT_FLAG_SYSTEM">
			<since>
				<version>12.0.0</version>
			</since>
			<synopsis>Raised when Asterisk is shutdown or restarted.</synopsis>
			<syntax>
				<parameter name="Shutdown">
					<para>Whether the shutdown is proceeding cleanly (all channels
					were hungup successfully) or uncleanly (channels will be
					terminated)</para>
					<enumlist>
						<enum name="Uncleanly"/>
						<enum name="Cleanly"/>
					</enumlist>
				</parameter>
				<parameter name="Restart">
					<para>Whether or not a restart will occur.</para>
					<enumlist>
						<enum name="True"/>
						<enum name="False"/>
					</enumlist>
				</parameter>
			</syntax>
		</managerEventInstance>
	</managerEvent>
 ***/

#ifndef AF_LOCAL
#define AF_LOCAL AF_UNIX
#define PF_LOCAL PF_UNIX
#endif

#define AST_MAX_CONNECTS 128
#define NUM_MSGS 64

/*! Displayed copyright tag */
#define COPYRIGHT_TAG "Copyright (C) 1999 - 2025, Sangoma Technologies Corporation and others."

/*! \brief Welcome message when starting a CLI interface */
#define WELCOME_MESSAGE \
    ast_verbose("Asterisk %s, " COPYRIGHT_TAG "\n" \
                "Created by Mark Spencer <markster@digium.com>\n" \
                "Asterisk comes with ABSOLUTELY NO WARRANTY; type 'core show warranty' for details.\n" \
                "This is free software, with components licensed under the GNU General Public\n" \
                "License version 2 and other licenses; you are welcome to redistribute it under\n" \
                "certain conditions. Type 'core show license' for details.\n" \
                "=========================================================================\n", ast_get_version()) \

static int ast_socket = -1;		/*!< UNIX Socket for allowing remote control */
static int ast_socket_is_sd = 0; /*!< Is socket activation responsible for ast_socket? */
static int ast_consock = -1;		/*!< UNIX Socket for controlling another asterisk */
pid_t ast_mainpid;
struct console {
	int fd;				/*!< File descriptor */
	int p[2];			/*!< Pipe */
	pthread_t t;			/*!< Thread of handler */
	int mute;			/*!< Is the console muted for logs */
	int uid;			/*!< Remote user ID. */
	int gid;			/*!< Remote group ID. */
	int levels[NUMLOGLEVELS];	/*!< Which log levels are enabled for the console */
	/*! Verbosity level of this console. */
	int option_verbose;
};

struct ast_atexit {
	void (*func)(void);
	int is_cleanup;
	AST_LIST_ENTRY(ast_atexit) list;
};

static AST_LIST_HEAD_STATIC(atexits, ast_atexit);

struct timeval ast_startuptime;
struct timeval ast_lastreloadtime;

static History *el_hist;
static EditLine *el;
static char *remotehostname;

struct console consoles[AST_MAX_CONNECTS];

static int ast_el_add_history(const char *);
static int ast_el_read_history(const char *);
static int ast_el_write_history(const char *);

static void ast_el_read_default_histfile(void);
static void ast_el_write_default_histfile(void);

static void asterisk_daemon(int isroot, const char *runuser, const char *rungroup);

static char *_argv[256];

typedef enum {
	/*! Normal operation */
	NOT_SHUTTING_DOWN,
	/*! Committed to shutting down.  Final phase */
	SHUTTING_DOWN_FINAL,
	/*! Committed to shutting down.  Initial phase */
	SHUTTING_DOWN,
	/*!
	 * Valid values for quit_handler() niceness below.
	 * These shutdown/restart levels can be cancelled.
	 *
	 * Remote console exit right now
	 */
	SHUTDOWN_FAST,
	/*! core stop/restart now */
	SHUTDOWN_NORMAL,
	/*! core stop/restart gracefully */
	SHUTDOWN_NICE,
	/*! core stop/restart when convenient */
	SHUTDOWN_REALLY_NICE
} shutdown_nice_t;

static shutdown_nice_t shuttingdown = NOT_SHUTTING_DOWN;

/*! Prevent new channel allocation for shutdown. */
static int shutdown_pending;

static int restartnow;
static pthread_t consolethread = AST_PTHREADT_NULL;
static pthread_t mon_sig_flags;
static int canary_pid = 0;
static char canary_filename[128];
static int multi_thread_safe;

static char randompool[256];

#ifdef HAVE_CAP
static cap_t child_cap;
#endif

static int sig_alert_pipe[2] = { -1, -1 };
static struct {
	 unsigned int need_reload:1;
	 unsigned int need_quit:1;
	 unsigned int need_quit_handler:1;
	 unsigned int need_el_end:1;
} sig_flags;

#if !defined(LOW_MEMORY)
struct thread_list_t {
	AST_RWLIST_ENTRY(thread_list_t) list;
	char *name;
	pthread_t id;
	int lwp;
};

static AST_RWLIST_HEAD_STATIC(thread_list, thread_list_t);

void ast_register_thread(char *name)
{
	struct thread_list_t *new = ast_calloc(1, sizeof(*new));

	if (!new)
		return;

	ast_assert(multi_thread_safe);
	new->id = pthread_self();
	new->lwp = ast_get_tid();
	new->name = name; /* steal the allocated memory for the thread name */
	AST_RWLIST_WRLOCK(&thread_list);
	AST_RWLIST_INSERT_HEAD(&thread_list, new, list);
	AST_RWLIST_UNLOCK(&thread_list);
}

void ast_unregister_thread(void *id)
{
	struct thread_list_t *x;

	AST_RWLIST_WRLOCK(&thread_list);
	AST_RWLIST_TRAVERSE_SAFE_BEGIN(&thread_list, x, list) {
		if ((void *) x->id == id) {
			AST_RWLIST_REMOVE_CURRENT(list);
			break;
		}
	}
	AST_RWLIST_TRAVERSE_SAFE_END;
	AST_RWLIST_UNLOCK(&thread_list);
	if (x) {
		ast_free(x->name);
		ast_free(x);
	}
}

/*! \brief Print the contents of a file */
static int print_file(int fd, char *desc, const char *filename)
{
	FILE *f;
	char c;
	if (!(f = fopen(filename, "r"))) {
		return -1;
	}
	ast_cli(fd, "%s", desc);
	while ((c = fgetc(f)) != EOF) {
		ast_cli(fd, "%c", c);
	}
	fclose(f);
	/* no need for trailing new line, the file already has one */
	return 0;
}

/*! \brief Give an overview of core settings */
static char *handle_show_settings(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	char buf[BUFSIZ];
	struct ast_tm tm;
	char eid_str[128];
	struct rlimit limits;
	char pbx_uuid[AST_UUID_STR_LEN];
#if defined(HAVE_EACCESS) || defined(HAVE_EUIDACCESS)
	char dir[PATH_MAX];
#endif

	switch (cmd) {
	case CLI_INIT:
		e->command = "core show settings";
		e->usage = "Usage: core show settings\n"
			   "       Show core misc settings";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	ast_eid_to_str(eid_str, sizeof(eid_str), &ast_eid_default);
	ast_pbx_uuid_get(pbx_uuid, sizeof(pbx_uuid));

	ast_cli(a->fd, "\nPBX Core settings\n");
	ast_cli(a->fd, "-----------------\n");
	ast_cli(a->fd, "  Version:                     %s\n", ast_get_version());
	ast_cli(a->fd, "  ABI related Build Options:   %s\n", S_OR(ast_get_build_opts(), "(none)"));
	ast_cli(a->fd, "  All Build Options:           %s\n", S_OR(ast_get_build_opts_all(), "(none)"));
	if (ast_option_maxcalls)
		ast_cli(a->fd, "  Maximum calls:               %d (Current %d)\n", ast_option_maxcalls, ast_active_channels());
	else
		ast_cli(a->fd, "  Maximum calls:               Not set\n");

	if (getrlimit(RLIMIT_NOFILE, &limits)) {
		ast_cli(a->fd, "  Maximum open file handles:   Error because of %s\n", strerror(errno));
	} else if (limits.rlim_cur == RLIM_INFINITY) {
		ast_cli(a->fd, "  Maximum open file handles:   Unlimited\n");
	} else if (limits.rlim_cur < ast_option_maxfiles) {
		ast_cli(a->fd, "  Maximum open file handles:   %d (is) %d (requested)\n", (int) limits.rlim_cur, ast_option_maxfiles);
	} else {
		ast_cli(a->fd, "  Maximum open file handles:   %d\n", (int) limits.rlim_cur);
	}

	ast_cli(a->fd, "  Root console verbosity:      %d\n", option_verbose);
	ast_cli(a->fd, "  Current console verbosity:   %d\n", ast_verb_console_get());
	ast_cli(a->fd, "  Debug level:                 %d\n", option_debug);
	ast_cli(a->fd, "  Trace level:                 %d\n", option_trace);
	ast_cli(a->fd, "  Dump core on crash:          %s\n", ast_opt_dump_core ? "Yes" : "No");
	print_file(a->fd, "  Core dump file:              ", "/proc/sys/kernel/core_pattern");
	ast_cli(a->fd, "  Maximum load average:        %lf\n", ast_option_maxload);
#if defined(HAVE_SYSINFO)
	ast_cli(a->fd, "  Minimum free memory:         %ld MB\n", option_minmemfree);
#endif
	if (ast_localtime(&ast_startuptime, &tm, NULL)) {
		ast_strftime(buf, sizeof(buf), "%H:%M:%S", &tm);
		ast_cli(a->fd, "  Startup time:                %s\n", buf);
	}
	if (ast_localtime(&ast_lastreloadtime, &tm, NULL)) {
		ast_strftime(buf, sizeof(buf), "%H:%M:%S", &tm);
		ast_cli(a->fd, "  Last reload time:            %s\n", buf);
	}
	ast_cli(a->fd, "  System:                      %s/%s built by %s on %s %s\n", ast_build_os, ast_build_kernel, ast_build_user, ast_build_machine, ast_build_date);
	ast_cli(a->fd, "  System name:                 %s\n", ast_config_AST_SYSTEM_NAME);
	ast_cli(a->fd, "  Entity ID:                   %s\n", eid_str);
	ast_cli(a->fd, "  PBX UUID:                    %s\n", pbx_uuid);
	ast_cli(a->fd, "  Default language:            %s\n", ast_defaultlanguage);
	ast_cli(a->fd, "  Language prefix:             %s\n", ast_language_is_prefix ? "Enabled" : "Disabled");
	ast_cli(a->fd, "  User name and group:         %s/%s\n", ast_config_AST_RUN_USER, ast_config_AST_RUN_GROUP);
#if defined(HAVE_EACCESS) || defined(HAVE_EUIDACCESS)
#if defined(HAVE_EUIDACCESS) && !defined(HAVE_EACCESS)
#define eaccess euidaccess
#endif
	if (!getcwd(dir, sizeof(dir))) {
		if (eaccess(dir, R_OK | X_OK | F_OK)) {
			ast_cli(a->fd, "  Running directory:           %s\n", "Unable to access");
		} else {
			ast_cli(a->fd, "  Running directory:           %s (%s)\n", dir, "Unable to access");
		}
	} else {
		ast_cli(a->fd, "  Running directory:           %s\n", dir);
	}
#endif /* defined(HAVE_EACCESS) || defined(HAVE_EUIDACCESS) */
	ast_cli(a->fd, "  Executable includes:         %s\n", ast_test_flag(&ast_options, AST_OPT_FLAG_EXEC_INCLUDES) ? "Enabled" : "Disabled");
	ast_cli(a->fd, "  Transcode via SLIN:          %s\n", ast_test_flag(&ast_options, AST_OPT_FLAG_TRANSCODE_VIA_SLIN) ? "Enabled" : "Disabled");
	ast_cli(a->fd, "  Transmit silence during rec: %s\n", ast_test_flag(&ast_options, AST_OPT_FLAG_TRANSMIT_SILENCE) ? "Enabled" : "Disabled");
	ast_cli(a->fd, "  Generic PLC:                 %s\n", ast_test_flag(&ast_options, AST_OPT_FLAG_GENERIC_PLC) ? "Enabled" : "Disabled");
	ast_cli(a->fd, "  Generic PLC on equal codecs: %s\n", ast_test_flag(&ast_options, AST_OPT_FLAG_GENERIC_PLC_ON_EQUAL_CODECS) ? "Enabled" : "Disabled");
	ast_cli(a->fd, "  Hide Msg Chan AMI events:    %s\n", ast_opt_hide_messaging_ami_events ? "Enabled" : "Disabled");
	ast_cli(a->fd, "  Sounds search custom dir:    %s\n", ast_opt_sounds_search_custom ? "Enabled" : "Disabled");
	ast_cli(a->fd, "  Min DTMF duration::          %u\n", option_dtmfminduration);
#if !defined(LOW_MEMORY)
	ast_cli(a->fd, "  Cache media frames:          %s\n", ast_opt_cache_media_frames ? "Enabled" : "Disabled");
#endif
	ast_cli(a->fd, "  RTP use dynamic payloads:    %u\n", ast_option_rtpusedynamic);

	if (ast_option_rtpptdynamic == AST_RTP_PT_LAST_REASSIGN) {
		ast_cli(a->fd, "  RTP dynamic payload types:   %u,%u-%u\n",
		        ast_option_rtpptdynamic,
		        AST_RTP_PT_FIRST_DYNAMIC, AST_RTP_MAX_PT - 1);
	} else if (ast_option_rtpptdynamic < AST_RTP_PT_LAST_REASSIGN) {
		ast_cli(a->fd, "  RTP dynamic payload types:   %u-%u,%u-%u\n",
		        ast_option_rtpptdynamic, AST_RTP_PT_LAST_REASSIGN,
		        AST_RTP_PT_FIRST_DYNAMIC, AST_RTP_MAX_PT - 1);
	} else {
		ast_cli(a->fd, "  RTP dynamic payload types:   %u-%u\n",
		        AST_RTP_PT_FIRST_DYNAMIC, AST_RTP_MAX_PT - 1);
	}
	ast_cli(a->fd, "  Shell on remote consoles:    %s\n",
		ast_option_disable_remote_console_shell ? "Disabled" : "Enabled");

	ast_cli(a->fd, "\n* Subsystems\n");
	ast_cli(a->fd, "  -------------\n");
	ast_cli(a->fd, "  Manager (AMI):               %s\n", ast_manager_check_enabled() ? "Enabled" : "Disabled");
	ast_cli(a->fd, "  Web Manager (AMI/HTTP):      %s\n", ast_webmanager_check_enabled() ? "Enabled" : "Disabled");
	ast_cli(a->fd, "  Call data records:           %s\n", ast_cdr_is_enabled() ? "Enabled" : "Disabled");
	ast_cli(a->fd, "  Realtime Architecture (ARA): %s\n", ast_realtime_enabled() ? "Enabled" : "Disabled");

	/*! \todo we could check musiconhold, voicemail, smdi, adsi, queues  */

	ast_cli(a->fd, "\n* Directories\n");
	ast_cli(a->fd, "  -------------\n");
	ast_cli(a->fd, "  Configuration file:          %s\n", ast_config_AST_CONFIG_FILE);
	ast_cli(a->fd, "  Configuration directory:     %s\n", ast_config_AST_CONFIG_DIR);
	ast_cli(a->fd, "  Module directory:            %s\n", ast_config_AST_MODULE_DIR);
	ast_cli(a->fd, "  Spool directory:             %s\n", ast_config_AST_SPOOL_DIR);
	ast_cli(a->fd, "  Log directory:               %s\n", ast_config_AST_LOG_DIR);
	ast_cli(a->fd, "  Run/Sockets directory:       %s\n", ast_config_AST_RUN_DIR);
	ast_cli(a->fd, "  PID file:                    %s\n", ast_config_AST_PID);
	ast_cli(a->fd, "  VarLib directory:            %s\n", ast_config_AST_VAR_DIR);
	ast_cli(a->fd, "  Data directory:              %s\n", ast_config_AST_DATA_DIR);
	ast_cli(a->fd, "  ASTDB:                       %s\n", ast_config_AST_DB);
	ast_cli(a->fd, "  IAX2 Keys directory:         %s\n", ast_config_AST_KEY_DIR);
	ast_cli(a->fd, "  AGI Scripts directory:       %s\n", ast_config_AST_AGI_DIR);
	ast_cli(a->fd, "  Cache directory:             %s\n", ast_config_AST_CACHE_DIR);
	ast_cli(a->fd, "\n\n");
	return CLI_SUCCESS;
}

static char *handle_show_threads(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int count = 0;
	struct thread_list_t *cur;
	switch (cmd) {
	case CLI_INIT:
		e->command = "core show threads";
		e->usage =
			"Usage: core show threads\n"
			"       List threads currently active in the system.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	AST_RWLIST_RDLOCK(&thread_list);
	AST_RWLIST_TRAVERSE(&thread_list, cur, list) {
		ast_cli(a->fd, "%p %d %s\n", (void *)cur->id, cur->lwp, cur->name);
		count++;
	}
	AST_RWLIST_UNLOCK(&thread_list);
	ast_cli(a->fd, "%d threads listed.\n", count);
	return CLI_SUCCESS;
}

#if defined (HAVE_SYSCTL) && defined(HAVE_SWAPCTL)
/*
 * swapmode is rewritten by Tobias Weingartner <weingart@openbsd.org>
 * to be based on the new swapctl(2) system call.
 */
static int swapmode(int *used, int *total)
{
	struct swapent *swdev;
	int nswap, rnswap, i;

	nswap = swapctl(SWAP_NSWAP, 0, 0);
	if (nswap == 0)
		return 0;

	swdev = ast_calloc(nswap, sizeof(*swdev));
	if (swdev == NULL)
		return 0;

	rnswap = swapctl(SWAP_STATS, swdev, nswap);
	if (rnswap == -1) {
		ast_free(swdev);
		return 0;
	}

	/* if rnswap != nswap, then what? */

	/* Total things up */
	*total = *used = 0;
	for (i = 0; i < nswap; i++) {
		if (swdev[i].se_flags & SWF_ENABLE) {
			*used += (swdev[i].se_inuse / (1024 / DEV_BSIZE));
			*total += (swdev[i].se_nblks / (1024 / DEV_BSIZE));
		}
	}
	ast_free(swdev);
	return 1;
}
#endif

#if defined(HAVE_SYSINFO) || defined(HAVE_SYSCTL)
/*! \brief Give an overview of system statistics */
static char *handle_show_sysinfo(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	uint64_t physmem, freeram;
#if defined(HAVE_SYSINFO) || defined(HAVE_SWAPCTL)
	int totalswap = 0;
	uint64_t freeswap = 0;
#endif
	int nprocs = 0;
	long uptime = 0;
#if defined(HAVE_SYSINFO)
	struct sysinfo sys_info;
#elif defined(HAVE_SYSCTL)
	static int pageshift;
	struct vmtotal vmtotal;
	struct timeval	boottime;
	time_t	now;
	int mib[2], pagesize;
#if defined(HAVE_SWAPCTL)
	int usedswap = 0;
#endif
	size_t len;
#endif

	switch (cmd) {
	case CLI_INIT:
		e->command = "core show sysinfo";
		e->usage =
			"Usage: core show sysinfo\n"
			"       List current system information.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

#if defined(HAVE_SYSINFO)
	sysinfo(&sys_info);
	uptime = sys_info.uptime / 3600;
	physmem = sys_info.totalram * sys_info.mem_unit;
	freeram = (sys_info.freeram * sys_info.mem_unit) / 1024;
	totalswap = (sys_info.totalswap * sys_info.mem_unit) / 1024;
	freeswap = (sys_info.freeswap * sys_info.mem_unit) / 1024;
	nprocs = sys_info.procs;
#elif defined(HAVE_SYSCTL)
	/* calculate the uptime by looking at boottime */
	time(&now);
	mib[0] = CTL_KERN;
	mib[1] = KERN_BOOTTIME;
	len = sizeof(boottime);
	if (sysctl(mib, 2, &boottime, &len, NULL, 0) != -1) {
		uptime = now - boottime.tv_sec;
	}
	uptime = uptime/3600;
	/* grab total physical memory  */
	mib[0] = CTL_HW;
#if defined(HW_PHYSMEM64)
	mib[1] = HW_PHYSMEM64;
#else
	mib[1] = HW_PHYSMEM;
#endif
	len = sizeof(physmem);
	sysctl(mib, 2, &physmem, &len, NULL, 0);

	pagesize = getpagesize();
	pageshift = 0;
	while (pagesize > 1) {
		pageshift++;
		pagesize >>= 1;
	}

	/* we only need the amount of log(2)1024 for our conversion */
	pageshift -= 10;

	/* grab vm totals */
	mib[0] = CTL_VM;
	mib[1] = VM_METER;
	len = sizeof(vmtotal);
	sysctl(mib, 2, &vmtotal, &len, NULL, 0);
	freeram = (vmtotal.t_free << pageshift);
	/* generate swap usage and totals */
#if defined(HAVE_SWAPCTL)
	swapmode(&usedswap, &totalswap);
	freeswap = (totalswap - usedswap);
#endif
	/* grab number of processes */
#if defined(__OpenBSD__)
	mib[0] = CTL_KERN;
	mib[1] = KERN_NPROCS;
	len = sizeof(nprocs);
	sysctl(mib, 2, &nprocs, &len, NULL, 0);
#endif
#endif

	ast_cli(a->fd, "\nSystem Statistics\n");
	ast_cli(a->fd, "-----------------\n");
	ast_cli(a->fd, "  System Uptime:             %ld hours\n", uptime);
	ast_cli(a->fd, "  Total RAM:                 %" PRIu64 " KiB\n", physmem / 1024);
	ast_cli(a->fd, "  Free RAM:                  %" PRIu64 " KiB\n", freeram);
#if defined(HAVE_SYSINFO)
	ast_cli(a->fd, "  Buffer RAM:                %" PRIu64 " KiB\n", ((uint64_t) sys_info.bufferram * sys_info.mem_unit) / 1024);
#endif
#if defined(HAVE_SYSINFO) || defined(HAVE_SWAPCTL)
	ast_cli(a->fd, "  Total Swap Space:          %d KiB\n", totalswap);
	ast_cli(a->fd, "  Free Swap Space:           %" PRIu64 " KiB\n\n", freeswap);
#endif
	ast_cli(a->fd, "  Number of Processes:       %d \n\n", nprocs);
	return CLI_SUCCESS;
}
#endif

struct profile_entry {
	const char *name;
	uint64_t	scale;	/* if non-zero, values are scaled by this */
	int64_t	mark;
	int64_t	value;
	int64_t	events;
};

struct profile_data {
	int entries;
	int max_size;
	struct profile_entry e[0];
};

static struct profile_data *prof_data;
#endif /* ! LOW_MEMORY */

/*! \brief allocates a counter with a given name and scale.
 * \return Returns the identifier of the counter.
 */
int ast_add_profile(const char *name, uint64_t scale)
{
#if !defined(LOW_MEMORY)
	int l = sizeof(struct profile_data);
	int n = 10;	/* default entries */

	if (prof_data == NULL) {
		prof_data = ast_calloc(1, l + n*sizeof(struct profile_entry));
		if (prof_data == NULL)
			return -1;
		prof_data->entries = 0;
		prof_data->max_size = n;
	}
	if (prof_data->entries >= prof_data->max_size) {
		void *p;
		n = prof_data->max_size + 20;
		p = ast_realloc(prof_data, l + n*sizeof(struct profile_entry));
		if (p == NULL)
			return -1;
		prof_data = p;
		prof_data->max_size = n;
	}
	n = prof_data->entries++;
	prof_data->e[n].name = ast_strdup(name);
	prof_data->e[n].value = 0;
	prof_data->e[n].events = 0;
	prof_data->e[n].mark = 0;
	prof_data->e[n].scale = scale;
	return n;
#else /* if defined(LOW_MEMORY) */
	return 0;
#endif
}

int64_t ast_profile(int i, int64_t delta)
{
#if !defined(LOW_MEMORY)
	if (!prof_data || i < 0 || i > prof_data->entries)	/* invalid index */
		return 0;
	if (prof_data->e[i].scale > 1)
		delta /= prof_data->e[i].scale;
	prof_data->e[i].value += delta;
	prof_data->e[i].events++;
	return prof_data->e[i].value;
#else /* if defined(LOW_MEMORY) */
	return 0;
#endif
}

#if !defined(LOW_MEMORY)
/* The RDTSC instruction was introduced on the Pentium processor and is not
 * implemented on certain clones, like the Cyrix 586. Hence, the previous
 * expectation of __i386__ was in error. */
#if defined ( __i686__) && (defined(__FreeBSD__) || defined(__NetBSD__) || defined(linux))
#if defined(__FreeBSD__)
#include <machine/cpufunc.h>
#elif defined(__NetBSD__) || defined(linux)
static __inline uint64_t
rdtsc(void)
{
	uint64_t rv;

	__asm __volatile(".byte 0x0f, 0x31" : "=A" (rv));
	return (rv);
}
#endif
#else	/* supply a dummy function on other platforms */
static __inline uint64_t
rdtsc(void)
{
	return 0;
}
#endif
#endif /* ! LOW_MEMORY */

int64_t ast_mark(int i, int startstop)
{
#if !defined(LOW_MEMORY)
	if (!prof_data || i < 0 || i > prof_data->entries) /* invalid index */
		return 0;
	if (startstop == 1)
		prof_data->e[i].mark = rdtsc();
	else {
		prof_data->e[i].mark = (rdtsc() - prof_data->e[i].mark);
		if (prof_data->e[i].scale > 1)
			prof_data->e[i].mark /= prof_data->e[i].scale;
		prof_data->e[i].value += prof_data->e[i].mark;
		prof_data->e[i].events++;
	}
	return prof_data->e[i].mark;
#else /* if defined(LOW_MEMORY) */
	return 0;
#endif
}

#if !defined(LOW_MEMORY)
#define DEFINE_PROFILE_MIN_MAX_VALUES min = 0; \
	max = prof_data->entries;\
	if  (a->argc > 3) { /* specific entries */ \
		if (isdigit(a->argv[3][0])) { \
			min = atoi(a->argv[3]); \
			if (a->argc == 5 && strcmp(a->argv[4], "-")) \
				max = atoi(a->argv[4]); \
		} else \
			search = a->argv[3]; \
	} \
	if (max > prof_data->entries) \
		max = prof_data->entries;

static char *handle_show_profile(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int i, min, max;
	const char *search = NULL;
	switch (cmd) {
	case CLI_INIT:
		e->command = "core show profile";
		e->usage = "Usage: core show profile\n"
			   "       show profile information";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (prof_data == NULL)
		return 0;

	DEFINE_PROFILE_MIN_MAX_VALUES;
	ast_cli(a->fd, "profile values (%d, allocated %d)\n-------------------\n",
		prof_data->entries, prof_data->max_size);
	ast_cli(a->fd, "%6s   %8s  %10s %12s %12s  %s\n", "ID", "Scale", "Events",
			"Value", "Average", "Name");
	for (i = min; i < max; i++) {
		struct profile_entry *entry = &prof_data->e[i];
		if (!search || strstr(entry->name, search))
		    ast_cli(a->fd, "%6d: [%8ld] %10ld %12lld %12lld  %s\n",
			i,
			(long)entry->scale,
			(long)entry->events, (long long)entry->value,
			(long long)(entry->events ? entry->value / entry->events : entry->value),
			entry->name);
	}
	return CLI_SUCCESS;
}

static char *handle_clear_profile(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int i, min, max;
	const char *search = NULL;
	switch (cmd) {
	case CLI_INIT:
		e->command = "core clear profile";
		e->usage = "Usage: core clear profile\n"
			   "       clear profile information";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (prof_data == NULL)
		return 0;

	DEFINE_PROFILE_MIN_MAX_VALUES;
	for (i= min; i < max; i++) {
		if (!search || strstr(prof_data->e[i].name, search)) {
			prof_data->e[i].value = 0;
			prof_data->e[i].events = 0;
		}
	}
	return CLI_SUCCESS;
}
#undef DEFINE_PROFILE_MIN_MAX_VALUES

#endif /* ! LOW_MEMORY */

int ast_pbx_uuid_get(char *pbx_uuid, int length)
{
	return ast_db_get("pbx", "UUID", pbx_uuid, length);
}

static void publish_fully_booted(void)
{
	struct ast_json *json_object;
	int uptime = 0;
	int lastreloaded = 0;
	struct timeval tmp;
	struct timeval curtime = ast_tvnow();

	if (ast_startuptime.tv_sec) {
		tmp = ast_tvsub(curtime, ast_startuptime);
		uptime = (int) tmp.tv_sec;
	}

	if (ast_lastreloadtime.tv_sec) {
		tmp = ast_tvsub(curtime, ast_lastreloadtime);
		lastreloaded = (int) tmp.tv_sec;
	}

	json_object = ast_json_pack("{s: s, s: i, s: i}",
			"Status", "Fully Booted",
			"Uptime", uptime,
			"LastReload", lastreloaded);
	ast_manager_publish_event("FullyBooted", EVENT_FLAG_SYSTEM, json_object);
	ast_json_unref(json_object);
}

static void ast_run_atexits(int run_cleanups)
{
	struct ast_atexit *ae;

	AST_LIST_LOCK(&atexits);
	while ((ae = AST_LIST_REMOVE_HEAD(&atexits, list))) {
		if (ae->func && (!ae->is_cleanup || run_cleanups)) {
			ae->func();
		}
		ast_free(ae);
	}
	AST_LIST_UNLOCK(&atexits);
}

static void __ast_unregister_atexit(void (*func)(void))
{
	struct ast_atexit *ae;

	AST_LIST_TRAVERSE_SAFE_BEGIN(&atexits, ae, list) {
		if (ae->func == func) {
			AST_LIST_REMOVE_CURRENT(list);
			ast_free(ae);
			break;
		}
	}
	AST_LIST_TRAVERSE_SAFE_END;
}

static int register_atexit(void (*func)(void), int is_cleanup)
{
	struct ast_atexit *ae;

	ae = ast_calloc(1, sizeof(*ae));
	if (!ae) {
		return -1;
	}
	ae->func = func;
	ae->is_cleanup = is_cleanup;

	AST_LIST_LOCK(&atexits);
	__ast_unregister_atexit(func);
	AST_LIST_INSERT_HEAD(&atexits, ae, list);
	AST_LIST_UNLOCK(&atexits);

	return 0;
}

int ast_register_atexit(void (*func)(void))
{
	return register_atexit(func, 0);
}

int ast_register_cleanup(void (*func)(void))
{
	return register_atexit(func, 1);
}

void ast_unregister_atexit(void (*func)(void))
{
	AST_LIST_LOCK(&atexits);
	__ast_unregister_atexit(func);
	AST_LIST_UNLOCK(&atexits);
}

/* Sending commands from consoles back to the daemon requires a terminating NULL */
static int fdsend(int fd, const char *s)
{
	return write(fd, s, strlen(s) + 1);
}

/* Sending messages from the daemon back to the display requires _excluding_ the terminating NULL */
static int fdprint(int fd, const char *s)
{
	return write(fd, s, strlen(s));
}

/*! \brief NULL handler so we can collect the child exit status */
static void _null_sig_handler(int sig)
{
}

static struct sigaction null_sig_handler = {
	.sa_handler = _null_sig_handler,
	.sa_flags = SA_RESTART,
};

static struct sigaction ignore_sig_handler = {
	.sa_handler = SIG_IGN,
};

AST_MUTEX_DEFINE_STATIC(safe_system_lock);
/*! \brief Keep track of how many threads are currently trying to wait*() on
 *  a child process
 */
static unsigned int safe_system_level = 0;
static struct sigaction safe_system_prev_handler;

void ast_replace_sigchld(void)
{
	unsigned int level;

	ast_mutex_lock(&safe_system_lock);
	level = safe_system_level++;

	/* only replace the handler if it has not already been done */
	if (level == 0) {
		sigaction(SIGCHLD, &null_sig_handler, &safe_system_prev_handler);
	}

	ast_mutex_unlock(&safe_system_lock);
}

void ast_unreplace_sigchld(void)
{
	unsigned int level;

	ast_mutex_lock(&safe_system_lock);

	/* Wrapping around here is an error */
	ast_assert(safe_system_level > 0);

	level = --safe_system_level;

	/* only restore the handler if we are the last one */
	if (level == 0) {
		sigaction(SIGCHLD, &safe_system_prev_handler, NULL);
	}

	ast_mutex_unlock(&safe_system_lock);
}

/*! \brief fork and perform other preparations for spawning applications */
static pid_t safe_exec_prep(int dualfork)
{
	pid_t pid;

#if defined(HAVE_WORKING_FORK) || defined(HAVE_WORKING_VFORK)
	ast_replace_sigchld();

#ifdef HAVE_WORKING_FORK
	pid = fork();
#else
	pid = vfork();
#endif

	if (pid == 0) {
#ifdef HAVE_CAP
		cap_set_proc(child_cap);
#endif
#ifdef HAVE_WORKING_FORK
		if (ast_opt_high_priority) {
			ast_set_priority(0);
		}
		/* Close file descriptors and launch system command */
		ast_close_fds_above_n(STDERR_FILENO);
#endif
		if (dualfork) {
#ifdef HAVE_WORKING_FORK
			pid = fork();
#else
			pid = vfork();
#endif
			if (pid < 0) {
				/* Second fork failed. */
				/* No logger available. */
				_exit(1);
			}

			if (pid > 0) {
				/* This is the first fork, exit so the reaper finishes right away. */
				_exit(0);
			}

			/* This is the second fork.  The first fork will exit immediately so
			 * Asterisk doesn't have to wait for completion.
			 * ast_safe_system("cmd &") would run in the background, but the '&'
			 * cannot be added with ast_safe_execvp, so we have to double fork.
			 */
		}
	}

	if (pid < 0) {
		ast_log(LOG_WARNING, "Fork failed: %s\n", strerror(errno));
	}
#else
	ast_log(LOG_WARNING, "Fork failed: %s\n", strerror(ENOTSUP));
	pid = -1;
#endif

	return pid;
}

/*! \brief wait for spawned application to complete and unreplace sigchld */
static int safe_exec_wait(pid_t pid)
{
	int res = -1;

#if defined(HAVE_WORKING_FORK) || defined(HAVE_WORKING_VFORK)
	if (pid > 0) {
		for (;;) {
			int status;

			res = waitpid(pid, &status, 0);
			if (res > -1) {
				res = WIFEXITED(status) ? WEXITSTATUS(status) : -1;
				break;
			}
			if (errno != EINTR) {
				break;
			}
		}
	}

	ast_unreplace_sigchld();
#endif

	return res;
}

int ast_safe_execvp(int dualfork, const char *file, char *const argv[])
{
	pid_t pid = safe_exec_prep(dualfork);

	if (pid == 0) {
		execvp(file, argv);
		_exit(1);
		/* noreturn from _exit */
	}

	return safe_exec_wait(pid);
}

int ast_safe_system(const char *s)
{
	pid_t pid = safe_exec_prep(0);

	if (pid == 0) {
		execl("/bin/sh", "/bin/sh", "-c", s, (char *) NULL);
		_exit(1);
		/* noreturn from _exit */
	}

	return safe_exec_wait(pid);
}

/*!
 * \brief enable or disable a logging level to a specified console
 */
void ast_console_toggle_loglevel(int fd, int level, int state)
{
	int x;

	if (level >= NUMLOGLEVELS) {
		level = NUMLOGLEVELS - 1;
	}

	for (x = 0;x < AST_MAX_CONNECTS; x++) {
		if (fd == consoles[x].fd) {
			/*
			 * Since the logging occurs when levels are false, set to
			 * flipped iinput because this function accepts 0 as off and 1 as on
			 */
			consoles[x].levels[level] = state ? 0 : 1;
			return;
		}
	}
}

/*!
 * \brief mute or unmute a console from logging
 */
void ast_console_toggle_mute(int fd, int silent)
{
	int x;
	for (x = 0;x < AST_MAX_CONNECTS; x++) {
		if (fd == consoles[x].fd) {
			if (consoles[x].mute) {
				consoles[x].mute = 0;
				if (!silent)
					ast_cli(fd, "Console is not muted anymore.\n");
			} else {
				consoles[x].mute = 1;
				if (!silent)
					ast_cli(fd, "Console is muted.\n");
			}
			return;
		}
	}
	ast_cli(fd, "Couldn't find remote console.\n");
}

/*!
 * \brief log the string to all attached network console clients
 */
static void ast_network_puts_mutable(const char *string, int level, int sublevel)
{
	int x;

	for (x = 0; x < AST_MAX_CONNECTS; ++x) {
		if (consoles[x].fd < 0
			|| consoles[x].mute
			|| consoles[x].levels[level]
			|| (level == __LOG_VERBOSE && consoles[x].option_verbose < sublevel)) {
			continue;
		}
		fdprint(consoles[x].p[1], string);
	}
}

/*!
 * \brief log the string to the root console, and all attached
 * network console clients
 */
void ast_console_puts_mutable(const char *string, int level)
{
	ast_console_puts_mutable_full(string, level, 0);
}

static int console_print(const char *s);

void ast_console_puts_mutable_full(const char *message, int level, int sublevel)
{
	/* Send to the root console */
	console_print(message);

	/* Wake up a poll()ing console */
	if (ast_opt_console && consolethread != AST_PTHREADT_NULL) {
		pthread_kill(consolethread, SIGURG);
	}

	/* Send to any network console clients */
	ast_network_puts_mutable(message, level, sublevel);
}

/*!
 * \brief write the string to all attached console clients
 */
static void ast_network_puts(const char *string)
{
	int x;

	for (x = 0; x < AST_MAX_CONNECTS; ++x) {
		if (consoles[x].fd < 0) {
			continue;
		}
		fdprint(consoles[x].p[1], string);
	}
}

/*!
 * \brief write the string to the root console, and all attached
 * network console clients
 */
void ast_console_puts(const char *string)
{
	/* Send to the root console */
	fputs(string, stdout);
	fflush(stdout);

	/* Send to any network console clients */
	ast_network_puts(string);
}

static pthread_t lthread;

/*!
 * \brief read() function supporting the reception of user credentials.
 *
 * \param fd Socket file descriptor.
 * \param buffer Receive buffer.
 * \param size 'buffer' size.
 * \param con Console structure to set received credentials
 * \retval -1 on error
 * \retval the number of bytes received on success.
 */
static int read_credentials(int fd, char *buffer, size_t size, struct console *con)
{
#if defined(SO_PEERCRED)
#ifdef HAVE_STRUCT_SOCKPEERCRED_UID
#define HAVE_STRUCT_UCRED_UID
	struct sockpeercred cred;
#else
	struct ucred cred;
#endif
	socklen_t len = sizeof(cred);
#endif
#if defined(HAVE_GETPEEREID)
	uid_t uid;
	gid_t gid;
#else
	int uid, gid;
#endif
	int result;

	result = read(fd, buffer, size);
	if (result < 0) {
		return result;
	}

#if defined(SO_PEERCRED) && (defined(HAVE_STRUCT_UCRED_UID) || defined(HAVE_STRUCT_UCRED_CR_UID))
	if (getsockopt(fd, SOL_SOCKET, SO_PEERCRED, &cred, &len)) {
		return result;
	}
#if defined(HAVE_STRUCT_UCRED_UID)
	uid = cred.uid;
	gid = cred.gid;
#else /* defined(HAVE_STRUCT_UCRED_CR_UID) */
	uid = cred.cr_uid;
	gid = cred.cr_gid;
#endif /* defined(HAVE_STRUCT_UCRED_UID) */

#elif defined(HAVE_GETPEEREID)
	if (getpeereid(fd, &uid, &gid)) {
		return result;
	}
#else
	return result;
#endif
	con->uid = uid;
	con->gid = gid;

	return result;
}

/* This is the thread running the remote console on the main process. */
static void *netconsole(void *vconsole)
{
	struct console *con = vconsole;
	char hostname[MAXHOSTNAMELEN] = "";
	char inbuf[512];
	char outbuf[512];
	const char * const end_buf = inbuf + sizeof(inbuf);
	char *start_read = inbuf;
	int res;
	struct pollfd fds[2];

	if (gethostname(hostname, sizeof(hostname)-1))
		ast_copy_string(hostname, "<Unknown>", sizeof(hostname));
	snprintf(outbuf, sizeof(outbuf), "%s/%ld/%s\n", hostname, (long)ast_mainpid, ast_get_version());
	fdprint(con->fd, outbuf);
	ast_verb_console_register(&con->option_verbose);
	for (;;) {
		fds[0].fd = con->fd;
		fds[0].events = POLLIN;
		fds[0].revents = 0;
		fds[1].fd = con->p[0];
		fds[1].events = POLLIN;
		fds[1].revents = 0;

		res = ast_poll(fds, 2, -1);
		if (res < 0) {
			if (errno != EINTR)
				ast_log(LOG_WARNING, "poll returned < 0: %s\n", strerror(errno));
			continue;
		}
		if (fds[0].revents) {
			int cmds_read, bytes_read;
			if ((bytes_read = read_credentials(con->fd, start_read, end_buf - start_read, con)) < 1) {
				break;
			}
			/* XXX This will only work if it is the first command, and I'm not sure fixing it is worth the effort. */
			if (strncmp(inbuf, "cli quit after ", 15) == 0) {
				ast_cli_command_multiple_full(con->uid, con->gid, con->fd, bytes_read - 15, inbuf + 15);
				break;
			}
			/* ast_cli_command_multiple_full will only process individual commands terminated by a
			 * NULL and not trailing partial commands. */
			if (!(cmds_read = ast_cli_command_multiple_full(con->uid, con->gid, con->fd, bytes_read + start_read - inbuf, inbuf))) {
				/* No commands were read. We either have a short read on the first command
				 * with space left, or a command that is too long */
				if (start_read + bytes_read < end_buf) {
					start_read += bytes_read;
				} else {
					ast_log(LOG_ERROR, "Command too long! Skipping\n");
					start_read = inbuf;
				}
				continue;
			}
			if (start_read[bytes_read - 1] == '\0') {
				/* The read ended on a command boundary, start reading again at the head of inbuf */
				start_read = inbuf;
				continue;
			}
			/* If we get this far, we have left over characters that have not been processed.
			 * Advance to the character after the last command read by ast_cli_command_multiple_full.
			 * We are guaranteed to have at least cmds_read NULLs */
			while (cmds_read-- && (start_read = strchr(start_read, '\0'))) {
				start_read++;
			}
			memmove(inbuf, start_read, end_buf - start_read);
			start_read = end_buf - start_read + inbuf;
		}
		if (fds[1].revents) {
			res = read_credentials(con->p[0], outbuf, sizeof(outbuf), con);
			if (res < 1) {
				ast_log(LOG_ERROR, "read returned %d\n", res);
				break;
			}
			res = write(con->fd, outbuf, res);
			if (res < 1)
				break;
		}
	}
	ast_verb_console_unregister();
	if (!ast_opt_hide_connect) {
		ast_verb(3, "Remote UNIX connection disconnected\n");
	}
	close(con->fd);
	close(con->p[0]);
	close(con->p[1]);
	con->fd = -1;

	return NULL;
}

static void *listener(void *unused)
{
	struct sockaddr_un sunaddr;
	int s;
	socklen_t len;
	int x;
	int poll_result;
	struct pollfd fds[1];

	for (;;) {
		if (ast_socket < 0) {
			return NULL;
		}
		fds[0].fd = ast_socket;
		fds[0].events = POLLIN;
		poll_result = ast_poll(fds, 1, -1);
		pthread_testcancel();
		if (poll_result < 0) {
			if (errno != EINTR) {
				ast_log(LOG_WARNING, "poll returned error: %s\n", strerror(errno));
			}
			continue;
		}
		len = sizeof(sunaddr);
		s = accept(ast_socket, (struct sockaddr *)&sunaddr, &len);
		if (s < 0) {
			if (errno != EINTR)
				ast_log(LOG_WARNING, "Accept returned %d: %s\n", s, strerror(errno));
		} else {
#if defined(SO_PASSCRED)
			int sckopt = 1;
			/* turn on socket credentials passing. */
			if (setsockopt(s, SOL_SOCKET, SO_PASSCRED, &sckopt, sizeof(sckopt)) < 0) {
				ast_log(LOG_WARNING, "Unable to turn on socket credentials passing\n");
				close(s);
			} else
#endif
			{
				for (x = 0; x < AST_MAX_CONNECTS; x++) {
					if (consoles[x].fd >= 0) {
						continue;
					}
					if (socketpair(AF_LOCAL, SOCK_STREAM, 0, consoles[x].p)) {
						ast_log(LOG_ERROR, "Unable to create pipe: %s\n", strerror(errno));
						fdprint(s, "Server failed to create pipe\n");
						close(s);
						break;
					}
					ast_fd_set_flags(consoles[x].p[1], O_NONBLOCK);
					consoles[x].mute = 1; /* Default is muted, we will un-mute if necessary */
					/* Default uid and gid to -2, so then in cli.c/cli_has_permissions() we will be able
					   to know if the user didn't send the credentials. */
					consoles[x].uid = -2;
					consoles[x].gid = -2;
					/* Server default of remote console verbosity level is OFF. */
					consoles[x].option_verbose = 0;
					consoles[x].fd = s;
					if (ast_pthread_create_detached_background(&consoles[x].t, NULL, netconsole, &consoles[x])) {
						consoles[x].fd = -1;
						ast_log(LOG_ERROR, "Unable to spawn thread to handle connection: %s\n", strerror(errno));
						close(consoles[x].p[0]);
						close(consoles[x].p[1]);
						fdprint(s, "Server failed to spawn thread\n");
						close(s);
					}
					break;
				}
				if (x >= AST_MAX_CONNECTS) {
					fdprint(s, "No more connections allowed\n");
					ast_log(LOG_WARNING, "No more connections allowed\n");
					close(s);
				} else if ((consoles[x].fd > -1) && (!ast_opt_hide_connect)) {
					ast_verb(3, "Remote UNIX connection\n");
				}
			}
		}
	}
	return NULL;
}

static int ast_makesocket(void)
{
	struct sockaddr_un sunaddr;
	int res;
	int x;
	uid_t uid = -1;
	gid_t gid = -1;

	for (x = 0; x < AST_MAX_CONNECTS; x++) {
		consoles[x].fd = -1;
	}

	if (ast_socket_is_sd) {
		ast_socket = ast_sd_get_fd_un(SOCK_STREAM, ast_config_AST_SOCKET);

		goto start_lthread;
	}

	unlink(ast_config_AST_SOCKET);
	ast_socket = socket(PF_LOCAL, SOCK_STREAM, 0);
	if (ast_socket < 0) {
		ast_log(LOG_WARNING, "Unable to create control socket: %s\n", strerror(errno));
		return -1;
	}
	memset(&sunaddr, 0, sizeof(sunaddr));
	sunaddr.sun_family = AF_LOCAL;
	ast_copy_string(sunaddr.sun_path, ast_config_AST_SOCKET, sizeof(sunaddr.sun_path));
	res = bind(ast_socket, (struct sockaddr *)&sunaddr, sizeof(sunaddr));
	if (res) {
		ast_log(LOG_WARNING, "Unable to bind socket to %s: %s\n", ast_config_AST_SOCKET, strerror(errno));
		close(ast_socket);
		ast_socket = -1;
		return -1;
	}
	res = listen(ast_socket, 2);
	if (res < 0) {
		ast_log(LOG_WARNING, "Unable to listen on socket %s: %s\n", ast_config_AST_SOCKET, strerror(errno));
		close(ast_socket);
		ast_socket = -1;
		return -1;
	}

start_lthread:
	if (ast_pthread_create_background(&lthread, NULL, listener, NULL)) {
		ast_log(LOG_WARNING, "Unable to create listener thread.\n");
		close(ast_socket);
		return -1;
	}

	if (ast_socket_is_sd) {
		/* owner/group/permissions are set by systemd, we might not even have access
		 * to socket file so leave it alone */
		return 0;
	}

	if (!ast_strlen_zero(ast_config_AST_CTL_OWNER)) {
		struct passwd *pw;
		if ((pw = getpwnam(ast_config_AST_CTL_OWNER)) == NULL)
			ast_log(LOG_WARNING, "Unable to find uid of user %s\n", ast_config_AST_CTL_OWNER);
		else
			uid = pw->pw_uid;
	}

	if (!ast_strlen_zero(ast_config_AST_CTL_GROUP)) {
		struct group *grp;
		if ((grp = getgrnam(ast_config_AST_CTL_GROUP)) == NULL)
			ast_log(LOG_WARNING, "Unable to find gid of group %s\n", ast_config_AST_CTL_GROUP);
		else
			gid = grp->gr_gid;
	}

	if (chown(ast_config_AST_SOCKET, uid, gid) < 0)
		ast_log(LOG_WARNING, "Unable to change ownership of %s: %s\n", ast_config_AST_SOCKET, strerror(errno));

	if (!ast_strlen_zero(ast_config_AST_CTL_PERMISSIONS)) {
		unsigned int p1;
		mode_t p;
		sscanf(ast_config_AST_CTL_PERMISSIONS, "%30o", &p1);
		p = p1;
		if ((chmod(ast_config_AST_SOCKET, p)) < 0)
			ast_log(LOG_WARNING, "Unable to change file permissions of %s: %s\n", ast_config_AST_SOCKET, strerror(errno));
	}

	return 0;
}

static int ast_tryconnect(void)
{
	struct sockaddr_un sunaddr;
	int res;
	ast_consock = socket(PF_LOCAL, SOCK_STREAM, 0);
	if (ast_consock < 0) {
		fprintf(stderr, "Unable to create socket: %s\n", strerror(errno));
		return 0;
	}
	memset(&sunaddr, 0, sizeof(sunaddr));
	sunaddr.sun_family = AF_LOCAL;
	ast_copy_string(sunaddr.sun_path, ast_config_AST_SOCKET, sizeof(sunaddr.sun_path));
	res = connect(ast_consock, (struct sockaddr *)&sunaddr, sizeof(sunaddr));
	if (res) {
		close(ast_consock);
		ast_consock = -1;
		return 0;
	} else
		return 1;
}

/*! \brief Urgent handler
 *
 * Called by soft_hangup to interrupt the poll, read, or other
 * system call.  We don't actually need to do anything though.
 * Remember: Cannot EVER ast_log from within a signal handler
 */
static void _urg_handler(int num)
{
	return;
}

static struct sigaction urg_handler = {
	.sa_handler = _urg_handler,
};

static void _hup_handler(int num)
{
	int save_errno = errno;

	if (restartnow) {
		if (el) {
			el_end(el);
		}
		execvp(_argv[0], _argv);
	}

	printf("Received HUP signal -- Reloading configs\n");
	sig_flags.need_reload = 1;
	if (ast_alertpipe_write(sig_alert_pipe)) {
		fprintf(stderr, "hup_handler: write() failed: %s\n", strerror(errno));
	}
	errno = save_errno;
}

static struct sigaction hup_handler = {
	.sa_handler = _hup_handler,
	.sa_flags = SA_RESTART,
};

static void _child_handler(int sig)
{
	/* Must not ever ast_log or ast_verbose within signal handler */
	int n, status, save_errno = errno;

	/*
	 * Reap all dead children -- not just one
	 */
	for (n = 0; waitpid(-1, &status, WNOHANG) > 0; n++)
		;
	if (n == 0 && option_debug)
		printf("Huh?  Child handler, but nobody there?\n");
	errno = save_errno;
}

static struct sigaction child_handler = {
	.sa_handler = _child_handler,
	.sa_flags = SA_RESTART,
};

/*! \brief Set an X-term or screen title */
static void set_title(char *text)
{
	if (getenv("TERM") && strstr(getenv("TERM"), "xterm"))
		fprintf(stdout, "\033]2;%s\007", text);
}

static void set_icon(char *text)
{
	if (getenv("TERM") && strstr(getenv("TERM"), "xterm"))
		fprintf(stdout, "\033]1;%s\007", text);
}

/*! \brief Check whether we were set to high(er) priority. */
static int has_priority(void)
{
	/* Neither of these calls should fail with these arguments. */
#ifdef __linux__
	/* For SCHED_OTHER, SCHED_BATCH and SCHED_IDLE, this will return
	 * 0. For the realtime priorities SCHED_RR and SCHED_FIFO, it
	 * will return something >= 1. */
	return sched_getscheduler(0);
#else
	/* getpriority() can return a value in -20..19 (or even -INF..20)
	 * where negative numbers are high priority. We don't bother
	 * checking errno. If the query fails and it returns -1, we'll
	 * assume that we're running at high prio; a safe assumption
	 * that will enable the resource starvation monitor (canary)
	 * just in case. */
	return (getpriority(PRIO_PROCESS, 0) < 0);
#endif
}

/*! \brief Set priority on all known threads. */
static int set_priority_all(int pri)
{
#if !defined(__linux__)
	/* The non-linux version updates the entire process prio. */
	return ast_set_priority(pri);
#elif defined(LOW_MEMORY)
	ast_log(LOG_WARNING, "Unable to enumerate all threads to update priority\n");
	return ast_set_priority(pri);
#else
	struct thread_list_t *cur;
	struct sched_param sched;
	char const *policy_str;
	int policy;

	memset(&sched, 0, sizeof(sched));
	if (pri) {
		policy = SCHED_RR;
		policy_str = "realtime";
		sched.sched_priority = 10;
	} else {
		policy = SCHED_OTHER;
		policy_str = "regular";
		sched.sched_priority = 0;
	}
	if (sched_setscheduler(getpid(), policy, &sched)) {
		ast_log(LOG_WARNING, "Unable to set %s thread priority on main thread\n", policy_str);
		return -1;
	}
	ast_verb(1, "Setting %s thread priority on all threads\n", policy_str);
	AST_RWLIST_RDLOCK(&thread_list);
	AST_RWLIST_TRAVERSE(&thread_list, cur, list) {
		/* Don't care about the return value. It should work. */
		sched_setscheduler(cur->lwp, policy, &sched);
	}
	AST_RWLIST_UNLOCK(&thread_list);
	return 0;
#endif
}

/*! \brief We set ourselves to a high priority, that we might pre-empt
 * everything else.  If your PBX has heavy activity on it, this is a
 * good thing.
 */
int ast_set_priority(int pri)
{
	struct sched_param sched;
	memset(&sched, 0, sizeof(sched));
#ifdef __linux__
	if (pri) {
		sched.sched_priority = 10;
		if (sched_setscheduler(0, SCHED_RR, &sched)) {
			return -1;
		}
	} else {
		sched.sched_priority = 0;
		/* According to the manpage, these parameters can never fail. */
		sched_setscheduler(0, SCHED_OTHER, &sched);
	}
#else
	if (pri) {
		if (setpriority(PRIO_PROCESS, 0, -10) == -1) {
			ast_log(LOG_WARNING, "Unable to set high priority\n");
			return -1;
		} else
			ast_verb(1, "Set to high priority\n");
	} else {
		/* According to the manpage, these parameters can never fail. */
		setpriority(PRIO_PROCESS, 0, 0);
	}
#endif
	return 0;
}

int ast_shutdown_final(void)
{
	return shuttingdown == SHUTTING_DOWN_FINAL;
}

int ast_shutting_down(void)
{
	return shutdown_pending;
}

int ast_cancel_shutdown(void)
{
	int shutdown_aborted = 0;

	ast_mutex_lock(&safe_system_lock);
	if (shuttingdown >= SHUTDOWN_FAST) {
		shuttingdown = NOT_SHUTTING_DOWN;
		shutdown_pending = 0;
		shutdown_aborted = 1;
	}
	ast_mutex_unlock(&safe_system_lock);
	return shutdown_aborted;
}

/*!
 * \internal
 * \brief Initiate system shutdown -- prevents new channels from being allocated.
 */
static void ast_begin_shutdown(void)
{
	ast_mutex_lock(&safe_system_lock);
	if (shuttingdown != NOT_SHUTTING_DOWN) {
		shutdown_pending = 1;
	}
	ast_mutex_unlock(&safe_system_lock);
}

static int can_safely_quit(shutdown_nice_t niceness, int restart);
static void really_quit(int num, shutdown_nice_t niceness, int restart);

static void quit_handler(int num, shutdown_nice_t niceness, int restart)
{
	if (can_safely_quit(niceness, restart)) {
		really_quit(num, niceness, restart);
		/* No one gets here. */
	}
	/* It wasn't our time. */
}

#define SHUTDOWN_TIMEOUT	15	/* Seconds */

/*!
 * \internal
 * \brief Wait for all channels to die, a timeout, or shutdown cancelled.
 * \since 13.3.0
 *
 * \param niceness Shutdown niceness in effect
 * \param seconds Number of seconds to wait or less than zero if indefinitely.
 *
 * \retval zero if waiting wasn't necessary.  We were idle.
 * \retval non-zero if we had to wait.
 */
static int wait_for_channels_to_die(shutdown_nice_t niceness, int seconds)
{
	time_t start;
	time_t now;
	int waited = 0;

	time(&start);
	for (;;) {
		if (!ast_undestroyed_channels() || shuttingdown != niceness) {
			break;
		}
		if (seconds < 0) {
			/* No timeout so just poll every second */
			sleep(1);
		} else {
			time(&now);

			/* Wait up to the given seconds for all channels to go away */
			if (seconds < (now - start)) {
				break;
			}

			/* Sleep 1/10 of a second */
			usleep(100000);
		}
		waited = 1;
	}
	return waited;
}

static int can_safely_quit(shutdown_nice_t niceness, int restart)
{
	int waited = 0;

	/* Check if someone else isn't already doing this. */
	ast_mutex_lock(&safe_system_lock);
	if (shuttingdown != NOT_SHUTTING_DOWN && niceness >= shuttingdown) {
		/* Already in progress and other request was less nice. */
		ast_mutex_unlock(&safe_system_lock);
		ast_verbose("Ignoring asterisk %s request, already in progress.\n", restart ? "restart" : "shutdown");
		return 0;
	}
	shuttingdown = niceness;
	ast_mutex_unlock(&safe_system_lock);

	/* Try to get as many CDRs as possible submitted to the backend engines
	 * (if in batch mode). really_quit happens to call it again when running
	 * the atexit handlers, otherwise this would be a bit early. */
	ast_cdr_engine_term();

	/*
	 * Shutdown the message queue for the technology agnostic message channel.
	 * This has to occur before we pause shutdown pending ast_undestroyed_channels.
	 *
	 * XXX This is not reversed on shutdown cancel.
	 */
	ast_msg_shutdown();

	if (niceness == SHUTDOWN_NORMAL) {
		/* Begin shutdown routine, hanging up active channels */
		ast_begin_shutdown();
		if (ast_opt_console) {
			ast_verb(0, "Beginning asterisk %s....\n", restart ? "restart" : "shutdown");
		}
		ast_softhangup_all();
		waited |= wait_for_channels_to_die(niceness, SHUTDOWN_TIMEOUT);
	} else if (niceness >= SHUTDOWN_NICE) {
		if (niceness != SHUTDOWN_REALLY_NICE) {
			ast_begin_shutdown();
		}
		if (ast_opt_console) {
			ast_verb(0, "Waiting for inactivity to perform %s...\n", restart ? "restart" : "halt");
		}
		waited |= wait_for_channels_to_die(niceness, -1);
	}

	/* Re-acquire lock and check if someone changed the niceness, in which
	 * case someone else has taken over the shutdown.
	 */
	ast_mutex_lock(&safe_system_lock);
	if (shuttingdown != niceness) {
		if (shuttingdown == NOT_SHUTTING_DOWN && ast_opt_console) {
			ast_verb(0, "Asterisk %s cancelled.\n", restart ? "restart" : "shutdown");
		}
		ast_mutex_unlock(&safe_system_lock);
		return 0;
	}

	if (niceness >= SHUTDOWN_REALLY_NICE) {
		shuttingdown = SHUTTING_DOWN;
		ast_mutex_unlock(&safe_system_lock);

		/* No more Mr. Nice guy.  We are committed to shutting down now. */
		ast_begin_shutdown();
		ast_softhangup_all();
		waited |= wait_for_channels_to_die(SHUTTING_DOWN, SHUTDOWN_TIMEOUT);

		ast_mutex_lock(&safe_system_lock);
	}
	shuttingdown = SHUTTING_DOWN_FINAL;
	ast_mutex_unlock(&safe_system_lock);

	if (niceness >= SHUTDOWN_NORMAL && waited) {
		/*
		 * We were not idle.  Give things in progress a chance to
		 * recognize the final shutdown phase.
		 */
		sleep(1);
	}
	return 1;
}

/*! Called when exiting is certain. */
static void really_quit(int num, shutdown_nice_t niceness, int restart)
{
	int active_channels;
	struct ast_json *json_object = NULL;
	int run_cleanups = niceness >= SHUTDOWN_NICE;

	if (run_cleanups && modules_shutdown()) {
		ast_verb(0, "Some modules could not be unloaded, switching to fast shutdown\n");
		run_cleanups = 0;
	}

	if (!restart && !ast_opt_remote) {
		ast_sd_notify("STOPPING=1");
	}
	if (ast_opt_console || (ast_opt_remote && !ast_opt_exec)) {
		ast_el_write_default_histfile();
		if (consolethread == AST_PTHREADT_NULL || consolethread == pthread_self()) {
			/* Only end if we are the consolethread, otherwise there's a race with that thread. */
			if (el != NULL) {
				el_end(el);
			}
			if (el_hist != NULL) {
				history_end(el_hist);
			}
		} else if (!restart) {
			sig_flags.need_el_end = 1;
			pthread_kill(consolethread, SIGURG);
		}
	}
	active_channels = ast_active_channels();
	/* Don't publish messages if we're a remote console - we won't have all of the Stasis
	 * topics or message types
	 */
	if (!ast_opt_remote) {
		json_object = ast_json_pack("{s: s, s: s}",
				"Shutdown", active_channels ? "Uncleanly" : "Cleanly",
				"Restart", restart ? "True" : "False");
		ast_manager_publish_event("Shutdown", EVENT_FLAG_SYSTEM, json_object);
		ast_json_unref(json_object);
		json_object = NULL;
	}
	ast_verb(0, "Asterisk %s ending (%d).\n",
		active_channels ? "uncleanly" : "cleanly", num);

	ast_verb(0, "Executing last minute cleanups\n");
	ast_run_atexits(run_cleanups);

	ast_debug(1, "Asterisk ending (%d).\n", num);
	if (ast_socket > -1) {
		pthread_cancel(lthread);
		close(ast_socket);
		ast_socket = -1;
		if (!ast_socket_is_sd) {
			unlink(ast_config_AST_SOCKET);
		}
		pthread_kill(lthread, SIGURG);
		pthread_join(lthread, NULL);
	}
	if (ast_consock > -1)
		close(ast_consock);
	if (!ast_opt_remote)
		unlink(ast_config_AST_PID);
	ast_alertpipe_close(sig_alert_pipe);
	printf("%s", term_quit());
	if (restart) {
		int i;
		ast_verb(0, "Preparing for Asterisk restart...\n");
		/* Mark all FD's for closing on exec */
		for (i = 3; i < 32768; i++) {
			fcntl(i, F_SETFD, FD_CLOEXEC);
		}
		ast_verb(0, "Asterisk is now restarting...\n");
		restartnow = 1;

		/* close logger */
		close_logger();
		clean_time_zones();

		/* If there is a consolethread running send it a SIGHUP
		   so it can execvp, otherwise we can do it ourselves */
		if ((consolethread != AST_PTHREADT_NULL) && (consolethread != pthread_self())) {
			pthread_kill(consolethread, SIGHUP);
			/* Give the signal handler some time to complete */
			sleep(2);
		} else
			execvp(_argv[0], _argv);

	} else {
		/* close logger */
		close_logger();
		clean_time_zones();
	}

	exit(0);
}

static void __quit_handler(int num)
{
	sig_flags.need_quit = 1;
	if (ast_alertpipe_write(sig_alert_pipe)) {
		fprintf(stderr, "quit_handler: write() failed: %s\n", strerror(errno));
	}
	/* There is no need to restore the signal handler here, since the app
	 * is going to exit */
}

static void __remote_quit_handler(int num)
{
	sig_flags.need_quit = 1;
}

static void set_header(char *outbuf, int maxout, char level)
{
	const char *cmp;
	char date[40];

	switch (level) {
	case 0: cmp = NULL;
		break;
	case 1: cmp = VERBOSE_PREFIX_1;
		break;
	case 2: cmp = VERBOSE_PREFIX_2;
		break;
	case 3: cmp = VERBOSE_PREFIX_3;
		break;
	case 4: cmp = VERBOSE_PREFIX_4;
		break;
	case 5: cmp = VERBOSE_PREFIX_5;
		break;
	case 6: cmp = VERBOSE_PREFIX_6;
		break;
	case 7: cmp = VERBOSE_PREFIX_7;
		break;
	case 8: cmp = VERBOSE_PREFIX_8;
		break;
	case 9: cmp = VERBOSE_PREFIX_9;
		break;
	default: cmp = VERBOSE_PREFIX_10;
		break;
	}

	if (ast_opt_timestamp) {
		struct ast_tm tm;
		struct timeval now = ast_tvnow();
		ast_localtime(&now, &tm, NULL);
		ast_strftime(date, sizeof(date), ast_logger_get_dateformat(), &tm);
	}

	snprintf(outbuf, maxout, "%s%s%s%s%s%s",
		ast_opt_timestamp ? "[" : "",
		ast_opt_timestamp ? date : "",
		ast_opt_timestamp ? "] " : "",
		cmp ? ast_term_color(COLOR_GRAY, 0) : "",
		cmp ? cmp : "",
		cmp ? ast_term_reset() : "");
}

struct console_state_data {
	char verbose_line_level;
};

static int console_state_init(void *ptr)
{
	struct console_state_data *state = ptr;
	state->verbose_line_level = 0;
	return 0;
}

AST_THREADSTORAGE_CUSTOM(console_state, console_state_init, ast_free_ptr);

static int console_print(const char *s)
{
	struct console_state_data *state =
		ast_threadstorage_get(&console_state, sizeof(*state));

	char prefix[80];
	const char *c;
	int num, res = 0;
	unsigned int newline;

	do {
		if (VERBOSE_HASMAGIC(s)) {

			/* always use the given line's level, otherwise
			   we'll use the last line's level */
			state->verbose_line_level = VERBOSE_MAGIC2LEVEL(s);

			/* move past magic */
			s++;

			set_header(prefix, sizeof(prefix), state->verbose_line_level);
		} else {
			*prefix = '\0';
		}
		c = s;

		/* for a given line separate on verbose magic, newline, and eol */
		if ((s = strchr(c, '\n'))) {
			++s;
			newline = 1;
		} else {
			s = strchr(c, '\0');
			newline = 0;
		}

		/* check if we should write this line after calculating begin/end
		   so we process the case of a higher level line embedded within
		   two lower level lines */
		if (state->verbose_line_level > option_verbose) {
			continue;
		}

		if (!ast_strlen_zero(prefix)) {
			fputs(prefix, stdout);
		}

		num = s - c;
		if (fwrite(c, sizeof(char), num, stdout) < num) {
			break;
		}

		if (!res) {
			/* if at least some info has been written
			   we'll want to return true */
			res = 1;
		}
	} while (*s);

	if (newline) {
		/* if ending on a newline then reset last level to zero
		    since what follows may be not be logging output */
		state->verbose_line_level = 0;
	}

	if (res) {
		fflush(stdout);
	}

	return res;
}

static int ast_all_zeros(const char *s)
{
	while (*s) {
		if (*s > 32)
			return 0;
		s++;
	}
	return 1;
}

/* This is the main console CLI command handler.  Run by the main() thread. */
static void consolehandler(const char *s)
{
	printf("%s", term_end());
	fflush(stdout);

	/* Called when readline data is available */
	if (!ast_all_zeros(s))
		ast_el_add_history(s);
	/* The real handler for bang */
	if (s[0] == '!') {
		if (s[1])
			ast_safe_system(s+1);
		else
			ast_safe_system(getenv("SHELL") ? getenv("SHELL") : "/bin/sh");
	} else
		ast_cli_command(STDOUT_FILENO, s);
}

static int remoteconsolehandler(const char *s)
{
	int ret = 0;

	/* Called when readline data is available */
	if (!ast_all_zeros(s))
		ast_el_add_history(s);

	while (isspace(*s)) {
		s++;
	}

	/* The real handler for bang */
	if (s[0] == '!') {
		if (ast_option_disable_remote_console_shell) {
			printf("Shell access is disabled on remote consoles\n");
			return 1;
		}
		if (s[1])
			ast_safe_system(s+1);
		else
			ast_safe_system(getenv("SHELL") ? getenv("SHELL") : "/bin/sh");
		ret = 1;
	} else if ((strncasecmp(s, "quit", 4) == 0 || strncasecmp(s, "exit", 4) == 0) &&
	    (s[4] == '\0' || isspace(s[4]))) {
		quit_handler(0, SHUTDOWN_FAST, 0);
		ret = 1;
	}

	return ret;
}

static char *handle_version(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "core show version";
		e->usage =
			"Usage: core show version\n"
			"       Shows Asterisk version information.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 3)
		return CLI_SHOWUSAGE;
	ast_cli(a->fd, "Asterisk %s built by %s @ %s on a %s running %s on %s\n",
		ast_get_version(), ast_build_user, ast_build_hostname,
		ast_build_machine, ast_build_os, ast_build_date);
	return CLI_SUCCESS;
}

static char *handle_stop_now(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "core stop now";
		e->usage =
			"Usage: core stop now\n"
			"       Shuts down a running Asterisk immediately, hanging up all active calls .\n";
		ast_cli_allow_at_shutdown(e);
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != e->args)
		return CLI_SHOWUSAGE;
	quit_handler(0, SHUTDOWN_NORMAL, 0 /* not restart */);
	return CLI_SUCCESS;
}

static char *handle_stop_gracefully(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "core stop gracefully";
		e->usage =
			"Usage: core stop gracefully\n"
			"       Causes Asterisk to not accept new calls, and exit when all\n"
			"       active calls have terminated normally.\n";
		ast_cli_allow_at_shutdown(e);
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != e->args)
		return CLI_SHOWUSAGE;
	quit_handler(0, SHUTDOWN_NICE, 0 /* no restart */);
	return CLI_SUCCESS;
}

static char *handle_stop_when_convenient(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "core stop when convenient";
		e->usage =
			"Usage: core stop when convenient\n"
			"       Causes Asterisk to perform a shutdown when all active calls have ended.\n";
		ast_cli_allow_at_shutdown(e);
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != e->args)
		return CLI_SHOWUSAGE;
	ast_cli(a->fd, "Waiting for inactivity to perform halt\n");
	quit_handler(0, SHUTDOWN_REALLY_NICE, 0 /* don't restart */);
	return CLI_SUCCESS;
}

static char *handle_restart_now(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "core restart now";
		e->usage =
			"Usage: core restart now\n"
			"       Causes Asterisk to hangup all calls and exec() itself performing a cold\n"
			"       restart.\n";
		ast_cli_allow_at_shutdown(e);
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != e->args)
		return CLI_SHOWUSAGE;
	quit_handler(0, SHUTDOWN_NORMAL, 1 /* restart */);
	return CLI_SUCCESS;
}

static char *handle_restart_gracefully(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "core restart gracefully";
		e->usage =
			"Usage: core restart gracefully\n"
			"       Causes Asterisk to stop accepting new calls and exec() itself performing a cold\n"
			"       restart when all active calls have ended.\n";
		ast_cli_allow_at_shutdown(e);
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != e->args)
		return CLI_SHOWUSAGE;
	quit_handler(0, SHUTDOWN_NICE, 1 /* restart */);
	return CLI_SUCCESS;
}

static char *handle_restart_when_convenient(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "core restart when convenient";
		e->usage =
			"Usage: core restart when convenient\n"
			"       Causes Asterisk to perform a cold restart when all active calls have ended.\n";
		ast_cli_allow_at_shutdown(e);
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != e->args)
		return CLI_SHOWUSAGE;
	ast_cli(a->fd, "Waiting for inactivity to perform restart\n");
	quit_handler(0, SHUTDOWN_REALLY_NICE, 1 /* restart */);
	return CLI_SUCCESS;
}

static char *handle_abort_shutdown(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "core abort shutdown";
		e->usage =
			"Usage: core abort shutdown\n"
			"       Causes Asterisk to abort an executing shutdown or restart, and resume normal\n"
			"       call operations.\n";
		ast_cli_allow_at_shutdown(e);
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != e->args)
		return CLI_SHOWUSAGE;

	ast_cancel_shutdown();

	return CLI_SUCCESS;
}

static char *handle_bang(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "!";
		e->usage =
			"Usage: !<command>\n"
			"       Executes a given shell command\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	return CLI_SUCCESS;
}
static const char warranty_lines[] = {
	"\n"
	"			    NO WARRANTY\n"
	"\n"
	"BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY\n"
	"FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT WHEN\n"
	"OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES\n"
	"PROVIDE THE PROGRAM \"AS IS\" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED\n"
	"OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF\n"
	"MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS\n"
	"TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE\n"
	"PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,\n"
	"REPAIR OR CORRECTION.\n"
	"\n"
	"IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING\n"
	"WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR\n"
	"REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES,\n"
	"INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING\n"
	"OUT OF THE USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED\n"
	"TO LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY\n"
	"YOU OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER\n"
	"PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE\n"
	"POSSIBILITY OF SUCH DAMAGES.\n"
};

static char *show_warranty(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "core show warranty";
		e->usage =
			"Usage: core show warranty\n"
			"       Shows the warranty (if any) for this copy of Asterisk.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	ast_cli(a->fd, "%s", warranty_lines);

	return CLI_SUCCESS;
}

static const char license_lines[] = {
	"\n"
	"This program is free software; you can redistribute it and/or modify\n"
	"it under the terms of the GNU General Public License version 2 as\n"
	"published by the Free Software Foundation.\n"
	"\n"
	"This program also contains components licensed under other licenses.\n"
	"They include:\n"
	"\n"
	"This program is distributed in the hope that it will be useful,\n"
	"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
	"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
	"GNU General Public License for more details.\n"
	"\n"
	"You should have received a copy of the GNU General Public License\n"
	"along with this program; if not, write to the Free Software\n"
	"Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA\n"
};

static char *show_license(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "core show license";
		e->usage =
			"Usage: core show license\n"
			"       Shows the license(s) for this copy of Asterisk.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	ast_cli(a->fd, "%s", license_lines);

	return CLI_SUCCESS;
}

#define ASTERISK_PROMPT "*CLI> "

/*!
 * \brief Shutdown Asterisk CLI commands.
 *
 * \note These CLI commands cannot be unregistered at shutdown
 * because one of them is likely the reason for the shutdown.
 * The CLI generates a warning if a command is in-use when it is
 * unregistered.
 */
static struct ast_cli_entry cli_asterisk_shutdown[] = {
	AST_CLI_DEFINE(handle_stop_now, "Shut down Asterisk immediately"),
	AST_CLI_DEFINE(handle_stop_gracefully, "Gracefully shut down Asterisk"),
	AST_CLI_DEFINE(handle_stop_when_convenient, "Shut down Asterisk at empty call volume"),
	AST_CLI_DEFINE(handle_restart_now, "Restart Asterisk immediately"),
	AST_CLI_DEFINE(handle_restart_gracefully, "Restart Asterisk gracefully"),
	AST_CLI_DEFINE(handle_restart_when_convenient, "Restart Asterisk at empty call volume"),
};

static struct ast_cli_entry cli_asterisk[] = {
	AST_CLI_DEFINE(handle_abort_shutdown, "Cancel a running shutdown"),
	AST_CLI_DEFINE(show_warranty, "Show the warranty (if any) for this copy of Asterisk"),
	AST_CLI_DEFINE(show_license, "Show the license(s) for this copy of Asterisk"),
	AST_CLI_DEFINE(handle_version, "Display version info"),
	AST_CLI_DEFINE(handle_bang, "Execute a shell command"),
#if !defined(LOW_MEMORY)
	AST_CLI_DEFINE(handle_show_threads, "Show running threads"),
#if defined(HAVE_SYSINFO) || defined(HAVE_SYSCTL)
	AST_CLI_DEFINE(handle_show_sysinfo, "Show System Information"),
#endif
	AST_CLI_DEFINE(handle_show_profile, "Display profiling info"),
	AST_CLI_DEFINE(handle_show_settings, "Show some core settings"),
	AST_CLI_DEFINE(handle_clear_profile, "Clear profiling info"),
#endif /* ! LOW_MEMORY */
};

static void send_rasterisk_connect_commands(void)
{
	char buf[80];

	/*
	 * Tell the server asterisk instance about the verbose level
	 * initially desired.
	 */
	if (option_verbose) {
		snprintf(buf, sizeof(buf), "core set verbose atleast %d silent", option_verbose);
		fdsend(ast_consock, buf);
	}

	if (option_debug) {
		snprintf(buf, sizeof(buf), "core set debug atleast %d", option_debug);
		fdsend(ast_consock, buf);
	}

	/* Leave verbose filtering to the server. */
	option_verbose = INT_MAX;

	if (!ast_opt_mute) {
		fdsend(ast_consock, "logger mute silent");
	} else {
		printf("log and verbose output currently muted ('logger mute' to unmute)\n");
	}
}

#ifdef HAVE_LIBEDIT_IS_UNICODE
#define CHAR_T_LIBEDIT wchar_t
#define CHAR_TO_LIBEDIT(c) btowc(c)
#else
#define CHAR_T_LIBEDIT char
#define CHAR_TO_LIBEDIT(c) c
#endif

static int ast_el_read_char(EditLine *editline, CHAR_T_LIBEDIT *cp)
{
	int num_read = 0;
	int lastpos = 0;
	struct pollfd fds[2];
	int res;
	int max;
#define EL_BUF_SIZE 512
	char buf[EL_BUF_SIZE];

	for (;;) {
		max = 1;
		fds[0].fd = ast_consock;
		fds[0].events = POLLIN;
		if (!ast_opt_exec) {
			fds[1].fd = STDIN_FILENO;
			fds[1].events = POLLIN;
			max++;
		}
		res = ast_poll(fds, max, -1);
		if (res < 0) {
			if (sig_flags.need_quit || sig_flags.need_quit_handler || sig_flags.need_el_end) {
				break;
			}
			if (errno == EINTR) {
				continue;
			}
			fprintf(stderr, "poll failed: %s\n", strerror(errno));
			break;
		}

		if (!ast_opt_exec && fds[1].revents) {
			char c = '\0';

			num_read = read(STDIN_FILENO, &c, 1);
			if (num_read < 1) {
				break;
			}

			*cp = CHAR_TO_LIBEDIT(c);

			return num_read;
		}

		if (fds[0].revents) {
			res = read(ast_consock, buf, sizeof(buf) - 1);
			/* if the remote side disappears exit */
			if (res < 1) {
				fprintf(stderr, "\nDisconnected from Asterisk server\n");
				if (!ast_opt_reconnect) {
					quit_handler(0, SHUTDOWN_FAST, 0);
				} else {
					int tries;
					int reconnects_per_second = 20;

					fprintf(stderr, "Attempting to reconnect for 30 seconds\n");
					for (tries = 0; tries < 30 * reconnects_per_second; tries++) {
						if (ast_tryconnect()) {
							fprintf(stderr, "Reconnect succeeded after %.3f seconds\n", 1.0 / reconnects_per_second * tries);
							printf("%s", term_quit());
							WELCOME_MESSAGE;
							send_rasterisk_connect_commands();
							break;
						}

						usleep(1000000 / reconnects_per_second);
					}
					if (tries >= 30 * reconnects_per_second) {
						fprintf(stderr, "Failed to reconnect for 30 seconds.  Quitting.\n");
						quit_handler(0, SHUTDOWN_FAST, 0);
					}
				}
				continue;
			}

			buf[res] = '\0';

			/* Write over the CLI prompt */
			if (!ast_opt_exec && !lastpos) {
				if (write(STDOUT_FILENO, "\r[0K", 5) < 0) {
				}
			}

			console_print(buf);

			if ((res < EL_BUF_SIZE - 1) && ((buf[res-1] == '\n') || (res >= 2 && buf[res-2] == '\n'))) {
				*cp = CHAR_TO_LIBEDIT(CC_REFRESH);

				return 1;
			}
			lastpos = 1;
		}
	}

	*cp = CHAR_TO_LIBEDIT('\0');

	return 0;
}

static struct ast_str *prompt = NULL;

static char *cli_prompt(EditLine *editline)
{
	char tmp[100];
	char *pfmt;
	int color_used = 0;
	static int cli_prompt_changes = 0;
	struct passwd *pw;
	struct group *gr;

	if (prompt == NULL) {
		prompt = ast_str_create(100);
	} else if (!cli_prompt_changes) {
		return ast_str_buffer(prompt);
	} else {
		ast_str_reset(prompt);
	}

	if ((pfmt = getenv("ASTERISK_PROMPT"))) {
		char *t = pfmt;
		struct timeval ts = ast_tvnow();
		while (*t != '\0') {
			if (*t == '%') {
				char hostname[MAXHOSTNAMELEN] = "";
				int i, which;
				struct ast_tm tm = { 0, };
				int fgcolor = COLOR_WHITE, bgcolor = COLOR_BLACK;

				t++;
				switch (*t) {
				case 'C': /* color */
					t++;
					if (sscanf(t, "%30d;%30d%n", &fgcolor, &bgcolor, &i) == 2) {
						ast_term_color_code(&prompt, fgcolor, bgcolor);
						t += i - 1;
					} else if (sscanf(t, "%30d%n", &fgcolor, &i) == 1) {
						ast_term_color_code(&prompt, fgcolor, 0);
						t += i - 1;
					}

					/* If the color has been reset correctly, then there's no need to reset it later */
					color_used = ((fgcolor == COLOR_WHITE) && (bgcolor == COLOR_BLACK)) ? 0 : 1;
					break;
				case 'd': /* date */
					if (ast_localtime(&ts, &tm, NULL)) {
						ast_strftime(tmp, sizeof(tmp), "%Y-%m-%d", &tm);
						ast_str_append(&prompt, 0, "%s", tmp);
						cli_prompt_changes++;
					}
					break;
				case 'g': /* group */
					if ((gr = getgrgid(getgid()))) {
						ast_str_append(&prompt, 0, "%s", gr->gr_name);
					}
					break;
				case 'h': /* hostname */
					if (!gethostname(hostname, sizeof(hostname) - 1)) {
						ast_str_append(&prompt, 0, "%s", hostname);
					} else {
						ast_str_append(&prompt, 0, "%s", "localhost");
					}
					break;
				case 'H': /* short hostname */
					if (!gethostname(hostname, sizeof(hostname) - 1)) {
						char *dotptr;
						if ((dotptr = strchr(hostname, '.'))) {
							*dotptr = '\0';
						}
						ast_str_append(&prompt, 0, "%s", hostname);
					} else {
						ast_str_append(&prompt, 0, "%s", "localhost");
					}
					break;
#ifdef HAVE_GETLOADAVG
				case 'l': /* load avg */
					t++;
					if (sscanf(t, "%30d", &which) == 1 && which > 0 && which <= 3) {
						double list[3];
						getloadavg(list, 3);
						ast_str_append(&prompt, 0, "%.2f", list[which - 1]);
						cli_prompt_changes++;
					}
					break;
#endif
				case 's': /* Asterisk system name (from asterisk.conf) */
					ast_str_append(&prompt, 0, "%s", ast_config_AST_SYSTEM_NAME);
					break;
				case 't': /* time */
					if (ast_localtime(&ts, &tm, NULL)) {
						ast_strftime(tmp, sizeof(tmp), "%H:%M:%S", &tm);
						ast_str_append(&prompt, 0, "%s", tmp);
						cli_prompt_changes++;
					}
					break;
				case 'u': /* username */
					if ((pw = getpwuid(getuid()))) {
						ast_str_append(&prompt, 0, "%s", pw->pw_name);
					}
					break;
				case '#': /* process console or remote? */
					ast_str_append(&prompt, 0, "%c", ast_opt_remote ? '>' : '#');
					break;
				case '%': /* literal % */
					ast_str_append(&prompt, 0, "%c", '%');
					break;
				case '\0': /* % is last character - prevent bug */
					t--;
					break;
				}
			} else {
				ast_str_append(&prompt, 0, "%c", *t);
			}
			t++;
		}
		if (color_used) {
			/* Force colors back to normal at end */
			ast_term_color_code(&prompt, 0, 0);
		}
	} else {
		ast_str_set(&prompt, 0, "%s%s",
			remotehostname ? remotehostname : "",
			ASTERISK_PROMPT);
	}

	return ast_str_buffer(prompt);
}

static struct ast_vector_string *ast_el_strtoarr(char *buf)
{
	char *retstr;
	struct ast_vector_string *vec = ast_calloc(1, sizeof(*vec));

	if (!vec) {
		return NULL;
	}

	while ((retstr = strsep(&buf, " "))) {
		if (!strcmp(retstr, AST_CLI_COMPLETE_EOF)) {
			break;
		}

		retstr = ast_strdup(retstr);
		if (!retstr || AST_VECTOR_APPEND(vec, retstr)) {
			ast_free(retstr);
			goto vector_cleanup;
		}
	}

	if (!AST_VECTOR_SIZE(vec)) {
		goto vector_cleanup;
	}

	return vec;

vector_cleanup:
	AST_VECTOR_CALLBACK_VOID(vec, ast_free);
	AST_VECTOR_PTR_FREE(vec);

	return NULL;
}

static void ast_cli_display_match_list(struct ast_vector_string *matches, int max)
{
	int idx = 1;
	/* find out how many entries can be put on one line, with two spaces between strings */
	int limit = ast_get_termcols(STDOUT_FILENO) / (max + 2);

	if (limit == 0) {
		limit = 1;
	}

	for (;;) {
		int numoutputline;

		for (numoutputline = 0; numoutputline < limit && idx < AST_VECTOR_SIZE(matches); idx++) {
			numoutputline++;
			fprintf(stdout, "%-*s  ", max, AST_VECTOR_GET(matches, idx));
		}

		if (!numoutputline) {
			break;
		}

		fprintf(stdout, "\n");
	}
}


static char *cli_complete(EditLine *editline, int ch)
{
	int len = 0;
	char *ptr;
	struct ast_vector_string *matches;
	int retval = CC_ERROR;
	char savechr;
	int res;

	LineInfo *lf = (LineInfo *)el_line(editline);

	savechr = *(char *)lf->cursor;
	*(char *)lf->cursor = '\0';
	ptr = (char *)lf->cursor;
	if (ptr) {
		while (ptr > lf->buffer) {
			if (isspace(*ptr)) {
				ptr++;
				break;
			}
			ptr--;
		}
	}

	len = lf->cursor - ptr;

	if (ast_opt_remote) {
#define CMD_MATCHESARRAY "_COMMAND MATCHESARRAY \"%s\" \"%s\""
		char *mbuf;
		char *new_mbuf;
		int mlen = 0;
		int maxmbuf = ast_asprintf(&mbuf, CMD_MATCHESARRAY, lf->buffer, ptr);

		if (maxmbuf == -1) {
			*((char *) lf->cursor) = savechr;

			return (char *)(CC_ERROR);
		}

		fdsend(ast_consock, mbuf);
		res = 0;
		mlen = 0;
		mbuf[0] = '\0';

		while (!strstr(mbuf, AST_CLI_COMPLETE_EOF) && res != -1) {
			if (mlen + 1024 > maxmbuf) {
				/* Expand buffer to the next 1024 byte increment plus a NULL terminator. */
				maxmbuf = mlen + 1024;
				new_mbuf = ast_realloc(mbuf, maxmbuf + 1);
				if (!new_mbuf) {
					ast_free(mbuf);
					*((char *) lf->cursor) = savechr;

					return (char *)(CC_ERROR);
				}
				mbuf = new_mbuf;
			}
			/* Only read 1024 bytes at a time */
			res = read(ast_consock, mbuf + mlen, 1024);
			if (res > 0) {
				if (!strncmp(mbuf, "Usage:", 6)) {
					/*
					 * Abort on malformed tab completes
					 * If help (tab complete) follows certain
					 * special characters, the main Asterisk process
					 * provides usage for the internal tab complete
					 * helper command that the remote console processes
					 * use.
					 * If this happens, the AST_CLI_COMPLETE_EOF sentinel
					 * value never gets sent. As a result, we'll just block
					 * forever if we don't handle this case.
					 * If we get command usage on a tab complete, then
					 * we know this scenario just happened and we should
					 * just silently ignore and do nothing.
					 */
					break;
				}
				mlen += res;
				mbuf[mlen] = '\0';
			}
		}
		mbuf[mlen] = '\0';

		matches = ast_el_strtoarr(mbuf);
		ast_free(mbuf);
	} else {
		matches = ast_cli_completion_vector((char *)lf->buffer, ptr);
	}

	if (matches) {
		int i;
		int maxlen, match_len;
		const char *best_match = AST_VECTOR_GET(matches, 0);

		if (!ast_strlen_zero(best_match)) {
			el_deletestr(editline, (int) len);
			el_insertstr(editline, best_match);
			retval = CC_REFRESH;
		}

		if (AST_VECTOR_SIZE(matches) == 2) {
			/* Found an exact match */
			el_insertstr(editline, " ");
			retval = CC_REFRESH;
		} else {
			/* Must be more than one match */
			for (i = 1, maxlen = 0; i < AST_VECTOR_SIZE(matches); i++) {
				match_len = strlen(AST_VECTOR_GET(matches, i));
				if (match_len > maxlen) {
					maxlen = match_len;
				}
			}

			fprintf(stdout, "\n");
			ast_cli_display_match_list(matches, maxlen);
			retval = CC_REDISPLAY;
		}
		AST_VECTOR_CALLBACK_VOID(matches, ast_free);
		AST_VECTOR_PTR_FREE(matches);
	}

	*((char *) lf->cursor) = savechr;

	return (char *)(long)retval;
}

static int ast_el_initialize(void)
{
	HistEvent ev;
	char *editor, *editrc = getenv("EDITRC");

	if (!(editor = getenv("AST_EDITMODE"))) {
		if (!(editor = getenv("AST_EDITOR"))) {
			editor = "emacs";
		}
	}

	if (el != NULL)
		el_end(el);
	if (el_hist != NULL)
		history_end(el_hist);

	el = el_init("asterisk", stdin, stdout, stderr);
	el_set(el, EL_PROMPT, cli_prompt);

	el_set(el, EL_EDITMODE, 1);
	el_set(el, EL_EDITOR, editor);
	el_hist = history_init();
	if (!el || !el_hist)
		return -1;

	/* setup history with 100 entries */
	history(el_hist, &ev, H_SETSIZE, 100);

	el_set(el, EL_HIST, history, el_hist);

	el_set(el, EL_ADDFN, "ed-complete", "Complete argument", cli_complete);
	/* Bind <tab> to command completion */
	el_set(el, EL_BIND, "^I", "ed-complete", NULL);
	/* Bind ? to command completion */
	el_set(el, EL_BIND, "?", "ed-complete", NULL);
	/* Bind ^D to redisplay */
	el_set(el, EL_BIND, "^D", "ed-redisplay", NULL);
	/* Bind Delete to delete char left */
	el_set(el, EL_BIND, "\\e[3~", "ed-delete-next-char", NULL);
	/* Bind Home and End to move to line start and end */
	el_set(el, EL_BIND, "\\e[1~", "ed-move-to-beg", NULL);
	el_set(el, EL_BIND, "\\e[4~", "ed-move-to-end", NULL);
	/* Bind C-left and C-right to move by word (not all terminals) */
	el_set(el, EL_BIND, "\\eOC", "vi-next-word", NULL);
	el_set(el, EL_BIND, "\\eOD", "vi-prev-word", NULL);

	if (editrc) {
		el_source(el, editrc);
	}

	return 0;
}

#define MAX_HISTORY_COMMAND_LENGTH 256

static int ast_el_add_history(const char *buf)
{
	HistEvent ev;
	char *stripped_buf;

	if (el_hist == NULL || el == NULL) {
		ast_el_initialize();
	}
	if (strlen(buf) > (MAX_HISTORY_COMMAND_LENGTH - 1)) {
		return 0;
	}

	stripped_buf = ast_strip(ast_strdupa(buf));

	/* HISTCONTROL=ignoredups */
	if (!history(el_hist, &ev, H_FIRST) && strcmp(ev.str, stripped_buf) == 0) {
		return 0;
	}

	return history(el_hist, &ev, H_ENTER, stripped_buf);
}

static int ast_el_write_history(const char *filename)
{
	HistEvent ev;

	if (el_hist == NULL || el == NULL)
		ast_el_initialize();

	return (history(el_hist, &ev, H_SAVE, filename));
}

static int ast_el_read_history(const char *filename)
{
	HistEvent ev;

	if (el_hist == NULL || el == NULL) {
		ast_el_initialize();
	}

	if (access(filename, F_OK) == 0) {
		return history(el_hist, &ev, H_LOAD, filename);
	}

	/* If the history file doesn't exist, failing to read it is unremarkable. */
	return 0;
}

static void process_histfile(int (*readwrite)(const char *filename))
{
	struct passwd *pw = getpwuid(geteuid());
	int ret = 0;
	char *name = NULL;

	if (!pw || ast_strlen_zero(pw->pw_dir)) {
		ast_log(LOG_ERROR, "Unable to determine home directory.  History read/write disabled.\n");
		return;
	}

	ret = ast_asprintf(&name, "%s/.asterisk_history", pw->pw_dir);
	if (ret <= 0) {
		ast_log(LOG_ERROR, "Unable to create history file name.  History read/write disabled.\n");
		return;
	}

	ret = readwrite(name);
	if (ret < 0) {
		ast_log(LOG_ERROR, "Unable to read or write history file '%s'\n", name);
	}

	ast_free(name);

	return;
}

static void ast_el_read_default_histfile(void)
{
	process_histfile(ast_el_read_history);
}

static void ast_el_write_default_histfile(void)
{
	process_histfile(ast_el_write_history);
}

static void ast_remotecontrol(char *data)
{
	char buf[256] = "";
	int res;
	char *hostname;
	char *cpid;
	char *version;
	int pid;
	char *stringp = NULL;

	char *ebuf;
	int num = 0;

	ast_term_init();
	printf("%s", term_end());
	fflush(stdout);

	memset(&sig_flags, 0, sizeof(sig_flags));
	signal(SIGINT, __remote_quit_handler);
	signal(SIGTERM, __remote_quit_handler);
	signal(SIGHUP, __remote_quit_handler);

	if (read(ast_consock, buf, sizeof(buf) - 1) < 0) {
		ast_log(LOG_ERROR, "read() failed: %s\n", strerror(errno));
		return;
	}
	if (data) {
		char prefix[] = "cli quit after ";
		char *tmp = ast_alloca(strlen(data) + strlen(prefix) + 1);
		sprintf(tmp, "%s%s", prefix, data);
		if (write(ast_consock, tmp, strlen(tmp) + 1) < 0) {
			ast_log(LOG_ERROR, "write() failed: %s\n", strerror(errno));
			if (sig_flags.need_quit || sig_flags.need_quit_handler || sig_flags.need_el_end) {
				return;
			}
		}
	}
	stringp = buf;
	hostname = strsep(&stringp, "/");
	cpid = strsep(&stringp, "/");
	version = strsep(&stringp, "\n");
	if (!version)
		version = "<Version Unknown>";
	stringp = hostname;
	strsep(&stringp, ".");
	if (cpid)
		pid = atoi(cpid);
	else
		pid = -1;
	if (!data) {
		send_rasterisk_connect_commands();
	}

	if (ast_opt_exec && data) {  /* hack to print output then exit if asterisk -rx is used */
		int linefull = 1, prev_linefull = 1, prev_line_verbose = 0;
		struct pollfd fds;
		fds.fd = ast_consock;
		fds.events = POLLIN;
		fds.revents = 0;

		while (ast_poll(&fds, 1, 60000) > 0) {
			char buffer[512] = "", *curline = buffer, *nextline;
			int not_written = 1;

			if (sig_flags.need_quit || sig_flags.need_quit_handler || sig_flags.need_el_end) {
				break;
			}

			if (read(ast_consock, buffer, sizeof(buffer) - 1) <= 0) {
				break;
			}

			do {
				prev_linefull = linefull;
				if ((nextline = strchr(curline, '\n'))) {
					linefull = 1;
					nextline++;
				} else {
					linefull = 0;
					nextline = strchr(curline, '\0');
				}

				/* Skip verbose lines */
				/* Prev line full? | Line is verbose | Last line verbose? | Print
				 * TRUE            | TRUE*           | TRUE               | FALSE
				 * TRUE            | TRUE*           | FALSE              | FALSE
				 * TRUE            | FALSE*          | TRUE               | TRUE
				 * TRUE            | FALSE*          | FALSE              | TRUE
				 * FALSE           | TRUE            | TRUE*              | FALSE
				 * FALSE           | TRUE            | FALSE*             | TRUE
				 * FALSE           | FALSE           | TRUE*              | FALSE
				 * FALSE           | FALSE           | FALSE*             | TRUE
				 */
				if ((!prev_linefull && !prev_line_verbose) || (prev_linefull && *curline > 0)) {
					prev_line_verbose = 0;
					not_written = 0;
					if (write(STDOUT_FILENO, curline, nextline - curline) < 0) {
						ast_log(LOG_WARNING, "write() failed: %s\n", strerror(errno));
					}
				} else {
					prev_line_verbose = 1;
				}
				curline = nextline;
			} while (!ast_strlen_zero(curline));

			/* No non-verbose output in 60 seconds. */
			if (not_written) {
				break;
			}
		}
		return;
	}

	ast_verbose("Connected to Asterisk %s currently running on %s (pid = %d)\n", version, hostname, pid);
	ast_init_logger_for_socket_console();

	remotehostname = hostname;
	if (el_hist == NULL || el == NULL)
		ast_el_initialize();
	ast_el_read_default_histfile();

	el_set(el, EL_GETCFN, ast_el_read_char);

	for (;;) {
		ebuf = (char *)el_gets(el, &num);

		if (sig_flags.need_quit || sig_flags.need_quit_handler || sig_flags.need_el_end) {
			break;
		}

		if (!ebuf && write(1, "", 1) < 0)
			break;

		if (!ast_strlen_zero(ebuf)) {
			if (ebuf[strlen(ebuf)-1] == '\n')
				ebuf[strlen(ebuf)-1] = '\0';
			if (!remoteconsolehandler(ebuf)) {
				res = write(ast_consock, ebuf, strlen(ebuf) + 1);
				if (res < 1) {
					ast_log(LOG_WARNING, "Unable to write: %s\n", strerror(errno));
					break;
				}
			}
		}
	}
	printf("\nDisconnected from Asterisk server\n");
}

static int show_version(void)
{
	printf("Asterisk %s\n", ast_get_version());
	return 0;
}

static int show_cli_help(void)
{
	printf("Asterisk %s, " COPYRIGHT_TAG "\n", ast_get_version());
	printf("Usage: asterisk [OPTIONS]\n");
	printf("Valid Options:\n");
	printf("   -V              Display version number and exit\n");
	printf("   -C <configfile> Use an alternate configuration file\n");
	printf("   -G <group>      Run as a group other than the caller\n");
	printf("   -U <user>       Run as a user other than the caller\n");
	printf("   -c              Provide console CLI\n");
	printf("   -d              Increase debugging (multiple d's = more debugging)\n");
#if HAVE_WORKING_FORK
	printf("   -f              Do not fork\n");
	printf("   -F              Always fork\n");
#endif
	printf("   -g              Dump core in case of a crash\n");
	printf("   -h              This help screen\n");
	printf("   -i              Initialize crypto keys at startup\n");
	printf("   -L <load>       Limit the maximum load average before rejecting new calls\n");
	printf("   -M <value>      Limit the maximum number of calls to the specified value\n");
	printf("   -m              Mute debugging and console output on the console\n");
	printf("   -n              Disable console colorization. Can be used only at startup.\n");
	printf("   -p              Run as pseudo-realtime thread\n");
	printf("   -q              Quiet mode (suppress output)\n");
	printf("   -r              Connect to Asterisk on this machine\n");
	printf("   -R              Same as -r, except attempt to reconnect if disconnected\n");
	printf("   -s <socket>     Connect to Asterisk via socket <socket> (only valid with -r)\n");
	printf("   -t              Record soundfiles in /var/tmp and move them where they\n");
	printf("                   belong after they are done\n");
	printf("   -T              Display the time in [Mmm dd hh:mm:ss] format for each line\n");
	printf("                   of output to the CLI. Cannot be used with remote console mode.\n\n");
	printf("   -v              Increase verbosity (multiple v's = more verbose)\n");
	printf("   -x <cmd>        Execute command <cmd> (implies -r)\n");
	printf("   -X              Enable use of #exec in asterisk.conf\n");
	printf("   -W              Adjust terminal colors to compensate for a light background\n");
	printf("\n");
	return 0;
}

static void read_pjproject_startup_options(void)
{
	struct ast_config *cfg;
	struct ast_variable *v;
	struct ast_flags config_flags = { CONFIG_FLAG_NOCACHE | CONFIG_FLAG_NOREALTIME };

	ast_option_pjproject_log_level = DEFAULT_PJ_LOG_MAX_LEVEL;
	ast_option_pjproject_cache_pools = DEFAULT_PJPROJECT_CACHE_POOLS;

	cfg = ast_config_load2("pjproject.conf", "" /* core, can't reload */, config_flags);
	if (!cfg
		|| cfg == CONFIG_STATUS_FILEUNCHANGED
		|| cfg == CONFIG_STATUS_FILEINVALID) {
		/* We'll have to use defaults */
		return;
	}

	for (v = ast_variable_browse(cfg, "startup"); v; v = v->next) {
		if (!strcasecmp(v->name, "log_level")) {
			if (sscanf(v->value, "%30d", &ast_option_pjproject_log_level) != 1) {
				ast_option_pjproject_log_level = DEFAULT_PJ_LOG_MAX_LEVEL;
			} else if (ast_option_pjproject_log_level < 0) {
				ast_option_pjproject_log_level = 0;
			} else if (MAX_PJ_LOG_MAX_LEVEL < ast_option_pjproject_log_level) {
				ast_option_pjproject_log_level = MAX_PJ_LOG_MAX_LEVEL;
			}
		} else if (!strcasecmp(v->name, "cache_pools")) {
			ast_option_pjproject_cache_pools = !ast_false(v->value);
		}
	}

	ast_config_destroy(cfg);
}

static void *monitor_sig_flags(void *unused)
{
	for (;;) {
		struct pollfd p = { sig_alert_pipe[0], POLLIN, 0 };

		ast_poll(&p, 1, -1);
		if (sig_flags.need_reload) {
			sig_flags.need_reload = 0;
			ast_module_reload(NULL);
		}
		if (sig_flags.need_quit) {
			sig_flags.need_quit = 0;
			if ((consolethread != AST_PTHREADT_NULL) && (consolethread != pthread_self())) {
				sig_flags.need_quit_handler = 1;
				pthread_kill(consolethread, SIGURG);
			} else {
				quit_handler(0, SHUTDOWN_NORMAL, 0);
			}
		}
		ast_alertpipe_read(sig_alert_pipe);
	}

	return NULL;
}

static void *canary_thread(void *unused)
{
	struct stat canary_stat;
	struct timeval now;

	/* Give the canary time to sing */
	sleep(120);

	for (;;) {
		now = ast_tvnow();
		if (stat(canary_filename, &canary_stat) || now.tv_sec > canary_stat.st_mtime + 60) {
			ast_log(LOG_WARNING,
				"The canary is no more.  He has ceased to be!  "
				"He's expired and gone to meet his maker!  "
				"He's a stiff!  Bereft of life, he rests in peace.  "
				"His metabolic processes are now history!  He's off the twig!  "
				"He's kicked the bucket.  He's shuffled off his mortal coil, "
				"run down the curtain, and joined the bleeding choir invisible!!  "
				"THIS is an EX-CANARY.  (Reducing priority)\n");
			set_priority_all(0);
			pthread_exit(NULL);
		}

		/* Check the canary once a minute */
		sleep(60);
	}
}

/* Used by libc's atexit(3) function */
static void canary_exit(void)
{
	if (canary_pid > 0) {
		int status;
		kill(canary_pid, SIGKILL);
		waitpid(canary_pid, &status, 0);
	}
}

enum startup_commands_phase {
	STARTUP_COMMANDS_PRE_INIT = 0,
	STARTUP_COMMANDS_PRE_MODULE,
	STARTUP_COMMANDS_FULLY_BOOTED
};

static const char *startup_commands_phase_str[] = {
	"pre-init",
	"pre-module",
	"fully-booted,yes,true,y,t,1,on"
};

/* Execute CLI commands on startup.  Run by main() thread. */
static void run_startup_commands(enum startup_commands_phase phase)
{
	int fd;
	struct ast_config *cfg;
	struct ast_flags cfg_flags = { 0 };
	struct ast_variable *v;

	if (!(cfg = ast_config_load2("cli.conf", "" /* core, can't reload */, cfg_flags)))
		return;
	if (cfg == CONFIG_STATUS_FILEMISSING || cfg == CONFIG_STATUS_FILEUNCHANGED || cfg == CONFIG_STATUS_FILEINVALID) {
		return;
	}

	fd = open("/dev/null", O_RDWR);
	if (fd < 0) {
		ast_config_destroy(cfg);
		return;
	}

	for (v = ast_variable_browse(cfg, "startup_commands"); v; v = v->next) {
		char *value = ast_str_to_lower(ast_strdupa(v->value));
		if (ast_in_delimited_string(value, startup_commands_phase_str[phase], ',')) {
			ast_cli_command(fd, v->name);
		}
	}

	close(fd);
	ast_config_destroy(cfg);
}

static void env_init(void)
{
	setenv("AST_SYSTEMNAME", ast_config_AST_SYSTEM_NAME, 1);
	setenv("AST_BUILD_HOST", ast_build_hostname, 1);
	setenv("AST_BUILD_DATE", ast_build_date, 1);
	setenv("AST_BUILD_KERNEL", ast_build_kernel, 1);
	setenv("AST_BUILD_MACHINE", ast_build_machine, 1);
	setenv("AST_BUILD_OS", ast_build_os, 1);
	setenv("AST_BUILD_USER", ast_build_user, 1);
	setenv("AST_VERSION", ast_get_version(), 1);
}

static void print_intro_message(const char *runuser, const char *rungroup)
{
 	if (ast_opt_console || option_verbose || (ast_opt_remote && !ast_opt_exec)) {
		WELCOME_MESSAGE;
		if (runuser) {
			ast_verbose("Running as user '%s'\n", runuser);
		}
		if (rungroup) {
			ast_verbose("Running under group '%s'\n", rungroup);
		}
	}
}

static void main_atexit(void)
{
	ast_cli_unregister_multiple(cli_asterisk, ARRAY_LEN(cli_asterisk));
}

int main(int argc, char *argv[])
{
	int c;
	int x;
	int isroot = 1, rundir_exists = 0;
	RAII_VAR(char *, runuser, NULL, ast_free);
	RAII_VAR(char *, rungroup, NULL, ast_free);
	RAII_VAR(char *, xarg, NULL, ast_free);
	struct rlimit l;
	static const char *getopt_settings = "BC:cde:FfG:ghIiL:M:mnpqRrs:TtU:VvWXx:";

	/* Remember original args for restart */
	if (argc > ARRAY_LEN(_argv) - 1) {
		fprintf(stderr, "Truncating argument size to %d\n", (int)ARRAY_LEN(_argv) - 1);
		argc = ARRAY_LEN(_argv) - 1;
	}
	for (x = 0; x < argc; x++)
		_argv[x] = argv[x];
	_argv[x] = NULL;

	if (geteuid() != 0)
		isroot = 0;

	/* if the progname is rasterisk consider it a remote console */
	if (argv[0] && (strstr(argv[0], "rasterisk")) != NULL) {
		ast_set_flag(&ast_options, AST_OPT_FLAG_NO_FORK | AST_OPT_FLAG_REMOTE);
	}
	ast_mainpid = getpid();

	/* Process command-line options that affect asterisk.conf load. */
	while ((c = getopt(argc, argv, getopt_settings)) != -1) {
		switch (c) {
		case 'X':
			ast_set_flag(&ast_options, AST_OPT_FLAG_EXEC_INCLUDES);
			break;
		case 'C':
			set_asterisk_conf_path(optarg);
			break;
		case 'd':
			option_debug++;
			break;
		case 'h':
			show_cli_help();
			exit(0);
		case 'R':
		case 'r':
		case 'x':
			/* ast_opt_remote is checked during config load.  This is only part of what
			 * these options do, see the second loop for the rest of the actions. */
			ast_set_flag(&ast_options, AST_OPT_FLAG_REMOTE);
			break;
		case 'V':
			show_version();
			exit(0);
		case 'v':
			option_verbose++;
			break;
		case '?':
			exit(1);
		}
	}

	/* Initialize env so it is available if #exec is used in asterisk.conf. */
	env_init();

	load_asterisk_conf();

	/* Update env to include any systemname that was set. */
	env_init();

	/*! \brief Check for options
	 *
	 * \todo Document these options
	 */
	optind = 1;
	while ((c = getopt(argc, argv, getopt_settings)) != -1) {
		/*!\note Please keep the ordering here to alphabetical, capital letters
		 * first.  This will make it easier in the future to select unused
		 * option flags for new features. */
		switch (c) {
		case 'B': /* Force black background */
			ast_set_flag(&ast_options, AST_OPT_FLAG_FORCE_BLACK_BACKGROUND);
			ast_clear_flag(&ast_options, AST_OPT_FLAG_LIGHT_BACKGROUND);
			break;
		case 'X':
			/* The command-line -X option enables #exec for asterisk.conf only. */
			break;
		case 'C':
			/* already processed. */
			break;
		case 'c':
			ast_set_flag(&ast_options, AST_OPT_FLAG_NO_FORK | AST_OPT_FLAG_CONSOLE);
			break;
		case 'd':
			/* already processed. */
			break;
#if defined(HAVE_SYSINFO)
		case 'e':
			if ((sscanf(&optarg[1], "%30ld", &option_minmemfree) != 1) || (option_minmemfree < 0)) {
				option_minmemfree = 0;
			}
			break;
#endif
#if HAVE_WORKING_FORK
		case 'F':
			ast_set_flag(&ast_options, AST_OPT_FLAG_ALWAYS_FORK);
			break;
		case 'f':
			ast_set_flag(&ast_options, AST_OPT_FLAG_NO_FORK);
			break;
#endif
		case 'G':
			rungroup = ast_strdup(optarg);
			break;
		case 'g':
			ast_set_flag(&ast_options, AST_OPT_FLAG_DUMP_CORE);
			break;
		case 'h':
			/* already processed. */
			break;
		case 'I':
			fprintf(stderr,
				"NOTICE: The -I option is no longer needed.\n"
				"  It will always be enabled if you have a timing module loaded.\n");
			break;
		case 'i':
			ast_set_flag(&ast_options, AST_OPT_FLAG_INIT_KEYS);
			break;
		case 'L':
			if ((sscanf(optarg, "%30lf", &ast_option_maxload) != 1) || (ast_option_maxload < 0.0)) {
				ast_option_maxload = 0.0;
			}
			break;
		case 'M':
			if ((sscanf(optarg, "%30d", &ast_option_maxcalls) != 1) || (ast_option_maxcalls < 0)) {
				ast_option_maxcalls = 0;
			}
			break;
		case 'm':
			ast_set_flag(&ast_options, AST_OPT_FLAG_MUTE);
			break;
		case 'n':
			ast_set_flag(&ast_options, AST_OPT_FLAG_NO_COLOR);
			break;
		case 'p':
			ast_set_flag(&ast_options, AST_OPT_FLAG_HIGH_PRIORITY);
			break;
		case 'q':
			ast_set_flag(&ast_options, AST_OPT_FLAG_QUIET);
			break;
		case 'R':
			ast_set_flag(&ast_options, AST_OPT_FLAG_NO_FORK | AST_OPT_FLAG_REMOTE | AST_OPT_FLAG_RECONNECT);
			break;
		case 'r':
			ast_set_flag(&ast_options, AST_OPT_FLAG_NO_FORK | AST_OPT_FLAG_REMOTE);
			break;
		case 's':
			if (ast_opt_remote) {
				set_socket_path(optarg);
			}
			break;
		case 'T':
			ast_set_flag(&ast_options, AST_OPT_FLAG_TIMESTAMP);
			break;
		case 't':
			ast_set_flag(&ast_options, AST_OPT_FLAG_CACHE_RECORD_FILES);
			break;
		case 'U':
			runuser = ast_strdup(optarg);
			break;
		case 'V':
		case 'v':
			/* already processed. */
			break;
		case 'W': /* White background */
			ast_set_flag(&ast_options, AST_OPT_FLAG_LIGHT_BACKGROUND);
			ast_clear_flag(&ast_options, AST_OPT_FLAG_FORCE_BLACK_BACKGROUND);
			break;
		case 'x':
			/* -r is implied by -x so set the flags -r sets as well. */
			ast_set_flag(&ast_options, AST_OPT_FLAG_NO_FORK | AST_OPT_FLAG_REMOTE);

			ast_set_flag(&ast_options, AST_OPT_FLAG_EXEC | AST_OPT_FLAG_NO_COLOR);
			xarg = ast_strdup(optarg);
			break;
		case '?':
			/* already processed. */
			break;
		}
	}

	if (ast_opt_remote) {
		int didwarn = 0;
		optind = 1;

		/* Not all options can be used with remote console. Warn if they're used. */
		while ((c = getopt(argc, argv, getopt_settings)) != -1) {
			switch (c) {
			/* okay to run with remote console */
			case 'B': /* force black background */
			case 'C': /* set config path */
			case 'd': /* debug */
			case 'h': /* help */
			case 'I': /* obsolete timing option: warning already thrown if used */
			case 'L': /* max load */
			case 'M': /* max calls */
			case 'm': /* mute */
			/*! \note The q option is never used anywhere, only defined */
			case 'q': /* quiet */
			case 'R': /* reconnect */
			case 'r': /* remote */
			/*! \note Can ONLY be used with remote console */
			case 's': /* set socket path */
			case 'T': /* timestamp */
			case 'V': /* version */
			case 'v': /* verbose */
			case 'W': /* white background */
			case 'x': /* remote execute */
			case '?': /* ? */
				break;
			/* can only be run when Asterisk is starting */
			case 'X': /* enables #exec for asterisk.conf only. */
			case 'c': /* foreground console */
			case 'e': /* minimum memory free */
			case 'F': /* always fork */
			case 'f': /* no fork */
			case 'G': /* run group */
			case 'g': /* dump core */
			case 'i': /* init keys */
			case 'n': /* no color */
			case 'p': /* high priority */
			case 't': /* cache record files */
			case 'U': /* run user */
				fprintf(stderr, "'%c' option is not compatible with remote console mode and has no effect.\n", c);
				didwarn = 1;
			}
		}
		if (didwarn) {
			fprintf(stderr, "\n"); /* if any warnings print out, make them stand out */
		}
	}

	/* For remote connections, change the name of the remote connection.
	 * We do this for the benefit of init scripts (which need to know if/when
	 * the main asterisk process has died yet). */
	if (ast_opt_remote) {
		strcpy(argv[0], "rasterisk");
		for (x = 1; x < argc; x++) {
			argv[x] = argv[0] + 10;
		}
	}

	if (!ast_language_is_prefix && !ast_opt_remote) {
		fprintf(stderr, "The 'languageprefix' option in asterisk.conf is deprecated; in a future release it will be removed, and your sound files will need to be organized in the 'new style' language layout.\n");
	}

	if (ast_opt_always_fork && (ast_opt_remote || ast_opt_console)) {
		fprintf(stderr, "'alwaysfork' is not compatible with console or remote console mode; ignored\n");
		ast_clear_flag(&ast_options, AST_OPT_FLAG_ALWAYS_FORK);
	}

	if (ast_opt_dump_core) {
		memset(&l, 0, sizeof(l));
		l.rlim_cur = RLIM_INFINITY;
		l.rlim_max = RLIM_INFINITY;
		if (setrlimit(RLIMIT_CORE, &l)) {
			fprintf(stderr, "Unable to disable core size resource limit: %s\n", strerror(errno));
		}
	}

	if (getrlimit(RLIMIT_NOFILE, &l)) {
		fprintf(stderr, "Unable to check file descriptor limit: %s\n", strerror(errno));
	}

#if !defined(CONFIGURE_RAN_AS_ROOT)
	/* Check if select(2) will run with more file descriptors */
	do {
		int fd, fd2;
		ast_fdset readers;
		struct timeval tv = { 0, };

		if (l.rlim_cur <= FD_SETSIZE) {
			/* The limit of select()able FDs is irrelevant, because we'll never
			 * open one that high. */
			break;
		}

		if (!(fd = open("/dev/null", O_RDONLY))) {
			fprintf(stderr, "Cannot open a file descriptor at boot? %s\n", strerror(errno));
			break; /* XXX Should we exit() here? XXX */
		}

		fd2 = ((l.rlim_cur > sizeof(readers) * 8) ? sizeof(readers) * 8 : l.rlim_cur) - 1;
		if (dup2(fd, fd2) < 0) {
			fprintf(stderr, "Cannot open maximum file descriptor %d at boot? %s\n", fd2, strerror(errno));
			close(fd);
			break;
		}

		FD_ZERO(&readers);
		FD_SET(fd2, &readers);
		if (ast_select(fd2 + 1, &readers, NULL, NULL, &tv) < 0) {
			fprintf(stderr, "Maximum select()able file descriptor is %d\n", FD_SETSIZE);
		}
		ast_FD_SETSIZE = l.rlim_cur > ast_FDMAX ? ast_FDMAX : l.rlim_cur;
		close(fd);
		close(fd2);
	} while (0);
#elif defined(HAVE_VARIABLE_FDSET)
	ast_FD_SETSIZE = l.rlim_cur > ast_FDMAX ? ast_FDMAX : l.rlim_cur;
#endif /* !defined(CONFIGURE_RAN_AS_ROOT) */

	if ((!rungroup) && !ast_strlen_zero(ast_config_AST_RUN_GROUP))
		rungroup = ast_strdup(ast_config_AST_RUN_GROUP);
	if ((!runuser) && !ast_strlen_zero(ast_config_AST_RUN_USER))
		runuser = ast_strdup(ast_config_AST_RUN_USER);

	/* Must install this signal handler up here to ensure that if the canary
	 * fails to execute that it doesn't kill the Asterisk process.
	 */
	sigaction(SIGCHLD, &child_handler, NULL);

	/* It's common on some platforms to clear /var/run at boot.  Create the
	 * socket file directory before we drop privileges. */
	if (mkdir(ast_config_AST_RUN_DIR, 0755)) {
		if (errno == EEXIST) {
			rundir_exists = 1;
		} else {
			fprintf(stderr, "Unable to create socket file directory.  Remote consoles will not be able to connect! (%s)\n", strerror(x));
		}
	}

#ifndef __CYGWIN__

	if (isroot) {
		ast_set_priority(ast_opt_high_priority);
	}

	if (isroot && rungroup) {
		struct group *gr;
		gr = getgrnam(rungroup);
		if (!gr) {
			fprintf(stderr, "No such group '%s'!\n", rungroup);
			exit(1);
		}
		if (!rundir_exists && chown(ast_config_AST_RUN_DIR, -1, gr->gr_gid)) {
			fprintf(stderr, "Unable to chgrp run directory to %d (%s)\n", (int) gr->gr_gid, rungroup);
		}
		if (setgid(gr->gr_gid)) {
			fprintf(stderr, "Unable to setgid to %d (%s)\n", (int)gr->gr_gid, rungroup);
			exit(1);
		}
		if (setgroups(0, NULL)) {
			fprintf(stderr, "Unable to drop unneeded groups\n");
			exit(1);
		}
	}

	if (runuser && !ast_test_flag(&ast_options, AST_OPT_FLAG_REMOTE)) {
#ifdef HAVE_CAP
		int has_cap = 1;
#endif /* HAVE_CAP */
		struct passwd *pw;
		pw = getpwnam(runuser);
		if (!pw) {
			fprintf(stderr, "No such user '%s'!\n", runuser);
			exit(1);
		}
		if (chown(ast_config_AST_RUN_DIR, pw->pw_uid, -1)) {
			fprintf(stderr, "Unable to chown run directory to %d (%s)\n", (int) pw->pw_uid, runuser);
		}
#ifdef HAVE_CAP
		if (prctl(PR_SET_KEEPCAPS, 1, 0, 0, 0)) {
			ast_log(LOG_WARNING, "Unable to keep capabilities.\n");
			has_cap = 0;
		}
#endif /* HAVE_CAP */
		if (!isroot && pw->pw_uid != geteuid()) {
			fprintf(stderr, "Asterisk started as nonroot, but runuser '%s' requested.\n", runuser);
			exit(1);
		}
		if (!rungroup) {
			if (setgid(pw->pw_gid)) {
				fprintf(stderr, "Unable to setgid to %d!\n", (int)pw->pw_gid);
				exit(1);
			}
			if (isroot && initgroups(pw->pw_name, pw->pw_gid)) {
				fprintf(stderr, "Unable to init groups for '%s'\n", runuser);
				exit(1);
			}
		}
		if (setuid(pw->pw_uid)) {
			fprintf(stderr, "Unable to setuid to %d (%s)\n", (int)pw->pw_uid, runuser);
			exit(1);
		}
#ifdef HAVE_CAP
		if (has_cap) {
			cap_t cap;

			cap = cap_from_text("cap_net_admin=eip");

			if (cap_set_proc(cap)) {
				fprintf(stderr, "Unable to install capabilities.\n");
			}
			if (cap_free(cap)) {
				fprintf(stderr, "Unable to drop capabilities.\n");
			}
		}
#endif /* HAVE_CAP */
	}

#endif /* __CYGWIN__ */

#ifdef linux
	if (geteuid() && ast_opt_dump_core) {
		if (prctl(PR_SET_DUMPABLE, 1, 0, 0, 0) < 0) {
			fprintf(stderr, "Unable to set the process for core dumps after changing to a non-root user. %s\n", strerror(errno));
		}
	}
#endif

	{
#if defined(HAVE_EACCESS) || defined(HAVE_EUIDACCESS)
		char dir[PATH_MAX];
		if (!getcwd(dir, sizeof(dir)) || eaccess(dir, R_OK | X_OK | F_OK)) {
			fprintf(stderr, "Unable to access the running directory (%s).  Changing to '/' for compatibility.\n", strerror(errno));
			/* If we cannot access the CWD, then we couldn't dump core anyway,
			 * so chdir("/") won't break anything. */
			if (chdir("/")) {
				/* chdir(/) should never fail, so this ends up being a no-op */
				fprintf(stderr, "chdir(\"/\") failed?!! %s\n", strerror(errno));
			}
		} else
#endif /* defined(HAVE_EACCESS) || defined(HAVE_EUIDACCESS) */
		if (!ast_opt_no_fork && !ast_opt_dump_core) {
			/* Backgrounding, but no cores, so chdir won't break anything. */
			if (chdir("/")) {
				fprintf(stderr, "Unable to chdir(\"/\") ?!! %s\n", strerror(errno));
			}
		}
	}

	/* Initial value of the maximum active system verbosity level. */
	ast_verb_sys_level = option_verbose;

	if (ast_sd_get_fd_un(SOCK_STREAM, ast_config_AST_SOCKET) > 0) {
		ast_socket_is_sd = 1;
	}

	/* DO NOT perform check for existing daemon if systemd has CLI socket activation */
	if (!ast_socket_is_sd && ast_tryconnect()) {
		/* One is already running */
		if (ast_opt_remote) {
			multi_thread_safe = 1;
			if (ast_opt_exec) {
				ast_remotecontrol(xarg);
				quit_handler(0, SHUTDOWN_FAST, 0);
				exit(0);
			}
			ast_term_init();
			printf("%s", term_end());
			fflush(stdout);

			print_intro_message(runuser, rungroup);
			printf("%s", term_quit());
			ast_remotecontrol(NULL);
			quit_handler(0, SHUTDOWN_FAST, 0);
			exit(0);
		} else {
			fprintf(stderr, "Asterisk already running on %s.  Use 'asterisk -r' to connect.\n", ast_config_AST_SOCKET);
			printf("%s", term_quit());
			exit(1);
		}
	} else if (ast_opt_remote || ast_opt_exec) {
		fprintf(stderr, "Unable to connect to remote asterisk (does %s exist?)\n", ast_config_AST_SOCKET);
		printf("%s", term_quit());
		exit(1);
	}

#ifdef HAVE_CAP
	child_cap = cap_from_text("cap_net_admin-eip");
#endif
	/* Not a remote console? Start the daemon. */
	asterisk_daemon(isroot, runuser, rungroup);
#ifdef HAS_CAP
	cap_free(child_cap);
#endif
	return 0;
}

static inline void check_init(int init_result, const char *name)
{
	if (init_result) {
		if (ast_is_logger_initialized()) {
			ast_log(LOG_ERROR, "%s initialization failed.  ASTERISK EXITING!\n%s", name, term_quit());
		} else {
			fprintf(stderr, "%s initialization failed.  ASTERISK EXITING!\n%s", name, term_quit());
		}
		ast_run_atexits(0);
		exit(init_result == -2 ? 2 : 1);
	}
}

static void asterisk_daemon(int isroot, const char *runuser, const char *rungroup)
{
	FILE *f;
	sigset_t sigs;
	int num;
	char *buf;
	char pbx_uuid[AST_UUID_STR_LEN];

	/* Set time as soon as possible */
	ast_lastreloadtime = ast_startuptime = ast_tvnow();

	/* This needs to remain as high up in the initial start up as possible.
	 * daemon causes a fork to occur, which has all sorts of unintended
	 * consequences for things that interact with threads.  This call *must*
	 * occur before anything in Asterisk spawns or manipulates thread related
	 * primitives. */
#if HAVE_WORKING_FORK
	if (ast_opt_always_fork || !ast_opt_no_fork) {
#ifndef HAVE_SBIN_LAUNCHD
		if (daemon(1, 0) < 0) {
			fprintf(stderr, "daemon() failed: %s\n", strerror(errno));
		} else {
			ast_mainpid = getpid();
		}
#else
		fprintf(stderr, "Mac OS X detected.  Use 'launchctl load /Library/LaunchDaemon/org.asterisk.asterisk.plist'.\n");
#endif
	}
#endif

	/* At this point everything has been forked successfully,
	 * we have determined that we aren't attempting to connect to
	 * an Asterisk instance, and that there isn't one already running. */
	multi_thread_safe = 1;

	load_astmm_phase_1();

	/* Check whether high prio was successfully set by us or some
	 * other incantation. */
	if (has_priority()) {
		ast_set_flag(&ast_options, AST_OPT_FLAG_HIGH_PRIORITY);
	} else {
		ast_clear_flag(&ast_options, AST_OPT_FLAG_HIGH_PRIORITY);
	}

	/* Spawning of astcanary must happen AFTER the call to daemon(3) */
	if (ast_opt_high_priority) {
		snprintf(canary_filename, sizeof(canary_filename), "%s/alt.asterisk.canary.tweet.tweet.tweet", ast_config_AST_RUN_DIR);

		/* Don't let the canary child kill Asterisk, if it dies immediately */
		sigaction(SIGPIPE, &ignore_sig_handler, NULL);

		canary_pid = fork();
		if (canary_pid == 0) {
			char canary_binary[PATH_MAX], ppid[12];

			/* Reset signal handler */
			signal(SIGCHLD, SIG_DFL);
			signal(SIGPIPE, SIG_DFL);

			ast_close_fds_above_n(0);
			ast_set_priority(0);
			snprintf(ppid, sizeof(ppid), "%d", (int) ast_mainpid);

			/* Use the astcanary binary that we installed */
			snprintf(canary_binary, sizeof(canary_binary), "%s/astcanary", ast_config_AST_SBIN_DIR);
			execl(canary_binary, "astcanary", canary_filename, ppid, (char *)NULL);

			/* Should never happen */
			_exit(1);
		} else if (canary_pid > 0) {
			pthread_t dont_care;
			ast_pthread_create_detached(&dont_care, NULL, canary_thread, NULL);
		}

		/* Kill the canary when we exit */
		ast_register_atexit(canary_exit);
	}

	/* Blindly write the PID file. */
	unlink(ast_config_AST_PID);
	f = fopen(ast_config_AST_PID, "w");
	if (f) {
		fprintf(f, "%ld\n", (long)ast_mainpid);
		fclose(f);
	} else {
		fprintf(stderr, "Unable to open pid file '%s': %s\n", ast_config_AST_PID, strerror(errno));
	}

	/* Initialize the terminal.  Since all processes have been forked,
	 * we can now start using the standard log messages.
	 */
	ast_term_init();
	printf("%s", term_end());
	fflush(stdout);

	print_intro_message(runuser, rungroup);

	register_config_cli();

	check_init(astobj2_init(), "AO2");
	check_init(ast_named_locks_init(), "Named Locks");

	if (ast_opt_console) {
		if (el_hist == NULL || el == NULL)
			ast_el_initialize();
		ast_el_read_default_histfile();
	}

#ifdef AST_XML_DOCS
	/* Load XML documentation. */
	ast_xmldoc_load_documentation();
#endif

	check_init(astdb_init(), "ASTdb");

	ast_uuid_init();

	if (ast_pbx_uuid_get(pbx_uuid, sizeof(pbx_uuid))) {
		ast_uuid_generate_str(pbx_uuid, sizeof(pbx_uuid));
		ast_db_put("pbx", "UUID", pbx_uuid);
	}
	ast_verb(0, "PBX UUID: %s\n", pbx_uuid);

	check_init(ast_json_init(), "libjansson");
	ast_ulaw_init();
	ast_alaw_init();
	ast_utf8_init();
	tdd_init();
	callerid_init();
	ast_builtins_init();

	check_init(ast_utils_init(), "Utilities");
	check_init(ast_tps_init(), "Task Processor Core");
	check_init(ast_fd_init(), "File Descriptor Debugging");
	check_init(ast_pbx_init(), "ast_pbx_init");
	check_init(aco_init(), "Configuration Option Framework");
	check_init(stasis_init(), "Stasis");
#ifdef TEST_FRAMEWORK
	check_init(ast_test_init(), "Test Framework");
#endif
	check_init(ast_translate_init(), "Translator Core");

	ast_aoc_cli_init();

	check_init(ast_sorcery_init(), "Sorcery");
	check_init(ast_codec_init(), "Codecs");
	check_init(ast_format_init(), "Formats");
	check_init(ast_format_cache_init(), "Format Cache");
	check_init(ast_codec_builtin_init(), "Built-in Codecs");
	check_init(ast_bucket_init(), "Bucket API");
	check_init(ast_stasis_system_init(), "Stasis system-level information");
	check_init(ast_endpoint_stasis_init(), "Stasis Endpoint");

	ast_makesocket();
	/* GCC 4.9 gives a bogus "right-hand operand of comma expression has
	 * no effect" warning */
	(void) sigemptyset(&sigs);
	(void) sigaddset(&sigs, SIGHUP);
	(void) sigaddset(&sigs, SIGTERM);
	(void) sigaddset(&sigs, SIGINT);
	(void) sigaddset(&sigs, SIGPIPE);
	(void) sigaddset(&sigs, SIGWINCH);
	pthread_sigmask(SIG_BLOCK, &sigs, NULL);
	sigaction(SIGURG, &urg_handler, NULL);
	signal(SIGINT, __quit_handler);
	signal(SIGTERM, __quit_handler);
	sigaction(SIGHUP, &hup_handler, NULL);
	sigaction(SIGPIPE, &ignore_sig_handler, NULL);

	/* ensure that the random number generators are seeded with a different value every time
	   Asterisk is started
	*/
	srand((unsigned int) getpid() + (unsigned int) time(NULL));
	initstate((unsigned int) getpid() * 65536 + (unsigned int) time(NULL), randompool, sizeof(randompool));

	threadstorage_init();

	check_init(init_logger(), "Logger");
	run_startup_commands(STARTUP_COMMANDS_PRE_INIT);

	check_init(ast_rtp_engine_init(), "RTP Engine");

	ast_autoservice_init();

	check_init(ast_timing_init(), "Timing");
	check_init(ast_ssl_init(), "SSL");
	read_pjproject_startup_options();
	check_init(ast_pj_init(), "Embedded PJProject");
	check_init(app_init(), "App Core");
	check_init(mwi_init(), "MWI Core");
	check_init(devstate_init(), "Device State Core");
	check_init(ast_msg_init(), "Messaging API");
	check_init(ast_channels_init(), "Channel");
	check_init(ast_endpoint_init(), "Endpoints");
	check_init(ast_pickup_init(), "Call Pickup");
	check_init(ast_bridging_init(), "Bridging");
	check_init(ast_parking_stasis_init(), "Parking Core");
	check_init(ast_device_state_engine_init(), "Device State Engine");
	check_init(ast_presence_state_engine_init(), "Presence State Engine");
	check_init(dns_core_init(), "DNS Resolver Core");
	check_init(ast_dns_system_resolver_init(), "Default DNS resolver");
	check_init(ast_security_stasis_init(), "Security Stasis Topic and Events");
	check_init(ast_image_init(), "Image");
	check_init(ast_file_init(), "Generic File Format Support");
	check_init(load_pbx(), "load_pbx");
	check_init(load_pbx_builtins(), "Builtin PBX Applications");
	check_init(load_pbx_functions_cli(), "PBX Functions Support");
	check_init(load_pbx_variables(), "PBX Variables Support");
	check_init(load_pbx_switch(), "PBX Switch Support");
	check_init(load_pbx_app(), "PBX Application Support");
	check_init(load_pbx_hangup_handler(), "PBX Hangup Handler Support");
	check_init(ast_local_init(), "Local Proxy Channel Driver");
	check_init(ast_refer_init(), "Refer API");

	run_startup_commands(STARTUP_COMMANDS_PRE_MODULE);

	/* We should avoid most config loads before this point as they can't use realtime. */
	check_init(load_modules(), "Module");

	/*
	 * This has to load after the dynamic modules load, as items in the media
	 * cache can't be constructed from items in the AstDB without their
	 * bucket backends.
	 */
	check_init(ast_media_cache_init(), "Media Cache");

	/* loads the cli_permissions.conf file needed to implement cli restrictions. */
	ast_cli_perms_init(0);
	ast_cli_channels_init(); /* Not always safe to access CLI commands until startup is complete. */

	ast_stun_init();

	dnsmgr_start_refresh();

	if (ast_opt_no_fork) {
		consolethread = pthread_self();
	}

	ast_alertpipe_init(sig_alert_pipe);

	ast_process_pending_reloads();

	ast_set_flag(&ast_options, AST_OPT_FLAG_FULLY_BOOTED);
	publish_fully_booted();

	pthread_sigmask(SIG_UNBLOCK, &sigs, NULL);

	load_astmm_phase_2();

	ast_cli_register_multiple(cli_asterisk_shutdown, ARRAY_LEN(cli_asterisk_shutdown));
	ast_cli_register_multiple(cli_asterisk, ARRAY_LEN(cli_asterisk));
	ast_register_cleanup(main_atexit);

	run_startup_commands(STARTUP_COMMANDS_FULLY_BOOTED);
	ast_sd_notify("READY=1");

	ast_verb(0, COLORIZE_FMT "\n", COLORIZE(COLOR_BRGREEN, 0, "Asterisk Ready."));

	logger_queue_start();

	if (ast_opt_console) {
		/* Console stuff now... */
		/* Register our quit function */
		char title[296];
		char hostname[MAXHOSTNAMELEN] = "";

		if (gethostname(hostname, sizeof(hostname) - 1)) {
			ast_copy_string(hostname, "<Unknown>", sizeof(hostname));
		}

		ast_pthread_create_detached(&mon_sig_flags, NULL, monitor_sig_flags, NULL);

		set_icon("Asterisk");
		snprintf(title, sizeof(title), "Asterisk Console on '%s' (pid %ld)", hostname, (long)ast_mainpid);
		set_title(title);

		el_set(el, EL_GETCFN, ast_el_read_char);

		for (;;) {
			if (sig_flags.need_el_end) {
				el_end(el);

				return;
			}

			if (sig_flags.need_quit || sig_flags.need_quit_handler) {
				quit_handler(0, SHUTDOWN_FAST, 0);
				break;
			}
			buf = (char *) el_gets(el, &num);

			if (!buf && write(1, "", 1) < 0)
				return; /* quit */

			if (buf) {
				if (buf[strlen(buf)-1] == '\n')
					buf[strlen(buf)-1] = '\0';

				consolehandler(buf);
			}
		}
	}

	/* Stall until a quit signal is given */
	monitor_sig_flags(NULL);
}
